<?php
/***********************************************/
/*               About This File               */
/***********************************************/
/*
	This file contains the shortcodes for the
	theme. Documentations for each shortcode
	should be made separately, based on the
	arguments defined here. Minimal inline
	documentation about arguments are provided
	here as well. The format of the argument
	documentation is:

	@argument [- required][- internal]: Description (possible values)

*/

/***********************************************/
/*              Table of Contents              */
/***********************************************/
/*

	1. Shortcodes
		1.1  Line
		1.2  Row
		1.3  Column
		1.4  Map
		1.5  Tabs
		1.6  Accordion
		1.7  Section
		1.8  Button
		1.9  Highlight
		1.10 Message
		1.11 Slideshow
		1.12 Postlist
		1.13 Sidebar

	2. Shortcode Helper Functions
		1.1 Shortcode Style Generator
*/

/***********************************************/
/*                 Shortcodes                  */
/***********************************************/

// 1.1 Line
/*
Outputs a separator line. Optionally a link
can be added to either side of the line.
@url: The url of the link
@text: The link text for the link
@position: The position (left, right)
@color: The color of the line (color)
*/
add_shortcode( 'line', COMPANYPREFIX . '_shortcode_line' );
function bsh_shortcode_line( $args ) {
	global $shortcode_style;
	$defaults = array(
		'url'         => '',
		'text'        => 'link',
		'position'    => 'left',
		'color'       => ''
	);
	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	$shortcode_style = '';
	bsh_shortcode_style( 'background-color', $color, 'color' );

	$output = '<div class="line primary-links" style="' . $shortcode_style . '">';
	if( !empty( $url ) ) {
		$output .= '<a class="align' . $position . '" href="' . $url . '">' . $text . '</a>';
	}
	$output .= '</div>';

	return $output;

}


// 1.3 Row
/*
A wrapper used for columns. Must be used together
with the [column] shortcode for it to work.
*/
add_shortcode( 'row', COMPANYPREFIX . '_shortcode_row' );
function bsh_shortcode_row( $args, $content ) {
	$output = '<div class="row user-columns">' . do_shortcode( $content ) . '</div>';
	return $output;
}

// 1.4 Column
/*
A shortcode to wrap column content. Must be used
together with the [row] shortcode
@width - required: The columns widths ( one - twelve )
*/
add_shortcode( 'column', COMPANYPREFIX . '_shortcode_column' );
function bsh_shortcode_column( $args, $content ) {
	$defaults = array(
		'width' => 'twelve',
	);
	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	$output = '<div class="' . $width . ' columns">' . do_shortcode( $content ) . '</div>';

	return $output;
}


// 1.5 Map
/*
Outputs a Google Map
@location - required: The location to center the map on
@type: The type of map shown (HYBRID, SATELLITE, ROADMAP, TERRAIN)
@zoom: The zoom level (1-18)
@marker: Weather or not a marker should be placed in the center (yes/no)
@height: The map height (400px)
*/
add_shortcode( 'map', COMPANYPREFIX . '_shortcode_map' );
function bsh_shortcode_map( $args ) {
	$defaults = array(
		'location' => '',
		'type'     => 'ROADMAP',
		'zoom'     => '14',
		'marker'   => 'yes',
		'height'   => '400px'
	);
	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	$id = rand( 9999, 999999 );
	ob_start();
	?>

	<div class="map-canvas" id='map-<?php echo $id ?>' style="width: 100%; height:<?php echo $height ?>"></div>
		<script type='text/javascript'>
		jQuery( document ).ready( function( $ ) {
			var geocoder = new google.maps.Geocoder();
			var address = '<?php echo $location  ?>';
			geocoder.geocode( { 'address': address}, function(results, status) {
				var lat = results[0].geometry.location.lat();
				var lng = results[0].geometry.location.lng();
				var latLng = new google.maps.LatLng(lat, lng);
				var mapOptions = {
				  center: latLng,
				  zoom: <?php echo $zoom ?>,
				  mapTypeId: google.maps.MapTypeId.<?php echo strtoupper($type) ?>
				};

				var map = new google.maps.Map( $("#map-<?php echo $id ?>")[0],
			    mapOptions);

			    <?php if( $marker == 'yes' ) : ?>
			    var marker = new google.maps.Marker({
			      position: latLng,
			      map: map,
			    });
			    <?php endif ?>
			})
		})

		</script>


	<?php
	$output = ob_get_clean();

	return $output;
}



// 1.6 Tabs
/*
Outputs a tabbed interface. Must be used together
with the [section] shortcode.
@contained: If set, the tab contents will be inside a box (yes/no)
@pill:  Use pill style tabs (yes/no)
*/
add_shortcode( 'tabs', COMPANYPREFIX . '_shortcode_tabs' );
function bsh_shortcode_tabs( $args, $content ) {
	$defaults = array(
		'contained'  => 'yes',
		'pill'       => 'no'
	);

	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	$contained = ( $contained == 'yes' ) ? 'contained' : '';
	$pill = ( $pill == 'yes' ) ? 'pill' : '';

	preg_match_all( "/\[section title=['|\"](.*?)['|\"]\](.*?)\[\/section\]/s", $content, $matches );
	$content_array = $matches[0];

	preg_match_all( "/\[section(.*)\]/", $content, $matches);
	$id = rand( 9999, 999999 );

	$output = '<dl class="tabs ' . $contained . ' ' . $pill . '">';
	$i = 0;
	foreach( $matches[1] as $tab ) {
		preg_match_all( "/([a-z]*?)=[\"|'](.*?)[\"|']/", $tab, $attr_matches, PREG_SET_ORDER );
		$tab_attr = array();
		foreach( $attr_matches as $attr ) {
			$tab_attr[$attr[1]] = $attr[2];
		}

		extract( $tab_attr );

		$activeClass = ( $i == 0 ) ? 'active' : '';
		$hideonmobile = ( !empty( $hideonmobile ) AND $hideonmobile == 'yes' ) ? 'hide-for-small' : '';
		$output .= '<dd class="' . $activeClass . ' ' . $hideonmobile . '"><a href="#tab-' . $id . '-' . $i . '">' . $title . '</a></dd>';


		$tabActive = ( $i==0 ) ? 'active="yes"' : '';

		$content_array[$i] = str_replace( '[section ', '[section type="tab" ' . $tabActive . '  id="tab-' . $id . '-' . $i . '" ', $content_array[$i] );
		$i++;
	}

	$content = implode( '', $content_array );
	$output .= '</dl><ul class="tabs-content ' . $contained . '">';
	$output .= do_shortcode( $content );
	$output .= '</ul>';

	return $output;

}


// 1.7 Accordion
/*
Outputs a collapsable accordion. Must be
used together with the [section] shortcode.
*/
add_shortcode( 'accordion',  COMPANYPREFIX . '_shortcode_accordion' );
function bsh_shortcode_accordion( $args, $content ) {
	$content = str_replace( '[section ', '[section type="accordion" ', $content );
	$pos = strpos( $content, '[section');
	$content_1 = substr( $content, 0, $pos + 8 );
	$content_2 = substr( $content, $pos + 8 );
	$content = $content_1 . ' active="yes"' . $content_2;

	$output = '<ul class="accordion">';
	$output .= do_shortcode( $content );
	$output .= '</ul>';
	return $output;
}

// 1.8 Section
/*
Used for creating accordion and tab sections.
Must be used together with the [accordion] or
the [tabs] shortcode.
@id - internal: A unique identifier
@active - internal: The active marker for the element
@type - internal: Used to dostinguish between types
@title - required: The title shown as the tab/accordion title
*/
add_shortcode( 'section', COMPANYPREFIX . '_shortcode_section' );
function bsh_shortcode_section( $args, $content ) {
	$defaults = array(
		'id'      => '',
		'active'  => '',
		'type'    => 'tab',
		'title'   => 'My Tab'
	);
	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	$activeClass = ( $active == 'yes' ) ? 'active' : '';

	if( $type == 'accordion' ) {
		$output = '<li class="' . $activeClass . '">
		    <div class="title">
		    <h5>' . $title . '</h5>
		    </div>
		    <div class="content">
		    ' . do_shortcode( $content ) . '
		    </div>
		</li>';
	}

	else {
		$output = '<li class="' . $activeClass . '" id="' . $id . 'Tab">' . do_shortcode( $content ) . '</li>';
	}

	return $output;

}

// 1.9 Button
/*
Used to create nicely formatted buttons
@url: The url to which the button points to
@background: The background color (color)
@color: The text color of the button (color)
@gradient: Adds or removes a gradient (no)
@radius: The amount of borde radius (+px)
@arrow: Weather or not to generate an arrow - uses the text color (yes/no)
*/
add_shortcode( 'button', COMPANYPREFIX . '_shortcode_button' );
function bsh_shortcode_button( $args, $content ) {
	global $shortcode_style;
	$defaults = array(
		'url'        => '',
		'background' => 'primary',
		'color'      => 'primary_text',
		'gradient'   => 'no',
		'radius'     => '5px',
		'arrow'      => 'no',
	);
	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	$shortcode_style = '';
	bsh_shortcode_style( 'background-color', $background, 'color' );
	if( $gradient === 'yes' ) {
		bsh_shortcode_style( 'background', $background, 'gradient' );
	}
	bsh_shortcode_style( 'border-color', $background, 'border-darken' );
	bsh_shortcode_style( 'color', $color, 'color' );
	if( !empty( $radius ) ) {
		bsh_shortcode_style( 'bordrer-radius', $radius, 'radius' );
	}

	$arrow_html = '';
	if( $arrow  == 'yes' ) {
		$arrow_html = '<span class="arrow"';
		if( !empty( $color ) ) {
			$arrow_html .= 'style="border-left:6px solid ' . $color . '"';
		}
		$arrow_html .= '></span>';
	}


	if( empty( $url ) ) {
		$output = '<span style="' . $shortcode_style . '" class="button">' . $content . $arrow_html .'</span>';
	}
	else {
		$output = '<a style="' . $shortcode_style . '" href="' . $url . '" class="button">' . $content . $arrow_html . '</a>';
	}

	return $output;
}

// 1.10 Highlight
/*
Used to highlight text
@background: The background color (color)
@color: The text color (color)
*/
add_shortcode( 'highlight', COMPANYPREFIX . '_shortcode_highlight' );
function bsh_shortcode_highlight( $args, $content ) {
	global $shortcode_style;
	$defaults = array(
		'background' => 'primary',
		'color'      => 'primary_text',
	);
	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	$shortcode_style = '';
	bsh_shortcode_style( 'background-color', $background, 'color' );
	bsh_shortcode_style( 'border-color', $background, 'border-darken' );
	bsh_shortcode_style( 'color', $color, 'color' );

	if( empty( $url ) ) {
		$output = '<span style="' . $shortcode_style . '" class="highlight">' . $content .'</span>';
	}
	else {
		$output = '<a style="' . $shortcode_style . '" class="highlight">' . $content . '</a>';
	}

	return $output;
}


// 1.11 Message
/*
Creates a callout style closable message
@background: The background color (color)
@color: The text color (color)
@radius: The amount of borde radius (+px)
*/
add_shortcode( 'message', COMPANYPREFIX . '_shortcode_message' );
function bsh_shortcode_message( $args, $content ) {
	global $shortcode_style;
	$defaults = array(
		'background'  => '',
		'color' => '',
		'radius' => '0px'
	);
	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	$shortcode_style = '';
	bsh_shortcode_style( 'background-color', $background, 'color' );
	bsh_shortcode_style( 'border-color', $background, 'border-darken' );
	bsh_shortcode_style( 'color', $color, 'color' );
	if( !empty( $radius ) ) {
		bsh_shortcode_style( 'border-radius', $radius, 'radius' );
	}

	$output = '<div style="' . $shortcode_style . '" class="alert-box message">' . do_shortcode($content) . ' <a href="" class="close">x</a></div>';

	return $output;

}


// 1.12 Slideshow
/*
Slideshows work exactly like the gallery shortcode
but pull the images into a nicely formatted slider
*/
add_shortcode( 'slideshow', COMPANYPREFIX . '_shortcode_slideshow' );
function bsh_shortcode_slideshow( $args ) {
	global $shortcode_style, $post;
	$defaults = array(
		'ids'         => '',
	);
	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	if( empty( $ids ) ) {
		$images = get_posts( 'post_type=attachment&post_mime_type=image&parent=' . $post->ID );
		foreach( $images as $image ) {
			$ids[] = $image->ID;
		}
	}
	else {
		$ids = explode( ',', $ids );
		$ids = array_map( 'trim', $ids );
	}

	$random = substr( sha1( time() . rand( 2234, 9999 ) ), 6 );

	$output = '<div class="post-slider flexslider" id="slider-' . $random . '"><ul class="slides">';
	foreach( $ids as $id ) {
		$image = wp_get_attachment_image( $id, bsh_get_layout_size( 'mus_slider' ) );
		$output .= '<li>' . $image . '</li>';
	}
	$output .= '</ul></div>';

	$output .= '
		<script type="text/javascript">
			$(window).load(function() {
				$("#slider-' . $random . '").flexslider({
					animation: "slide"
				});
			});
		</script>
	';

	return $output;

}

// 1.13 Postlist
/*
Postlists allow you to display a list of posts inside
the contents of another post
*/
add_shortcode( 'postlist', COMPANYPREFIX . '_shortcode_postlist' );
function bsh_shortcode_postlist( $args ) {
	global $shortcode_style, $post, $wp_query;
	$defaults = array(
		'post_type'         => 'post',
		'post_status'       => 'publish',
		'count'             => '3',
		'category'          => '',
		'orderby'           => 'post_date',
		'order'             => 'DESC',
		'type'              => 'future',
		'offset'            => 0,
		'layout'            => 'item',
	);
	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	$temp_query = $wp_query;
	$wp_query = null;

	$atts = array(
		'post_type'      => $post_type,
		'post_status'    => $post_status,
		'posts_per_page' => $count,
		'offset'         => $offset
	);

	if( !empty( $category ) ) {
		$categories = explode( ',', $category );
		$categories = array_map( 'trim', $categories );

		$atts['tax_query'] = array(
			array(
				'taxonomy' => 'category',
				'field' => 'id',
				'terms' => $categories,
				'operator' => 'IN'
			)
		);
	}


	if( $orderby == 'event_date' ) {
		$atts['orderby'] = 'meta_value';
		$atts['meta_key'] = '_mus_time';
	}
	else {
		$atts['orderby'] = $orderby;
	}

	$atts['order'] = $order;

	if( $type == 'future' AND $orderby == 'event_date' ) {
		$atts['meta_query'][] = array(
			'meta_key' => '_mus_time',
			'value'    => date( 'Y-m-d H:i:s' ),
			'compare'  => '>=',
			'type'     => 'DATETIME'
		);
	}
	elseif( $type == 'past' AND $orderby == 'event_date' ) {
		$atts['meta_query'][] = array(
			'meta_key' => '_mus_time',
			'value'    => date( 'Y-m-d H:i:s' ),
			'compare'  => '<=',
			'type'     => 'DATETIME'
		);
	}


	$wp_query = new WP_Query( $atts );



	$output = '';

	ob_start();
	if( have_posts() ) {
		echo '<div class="content-postlist">';
		while( have_posts() ) {
			the_post();
			get_template_part( 'layouts/layout', $post->post_type . '-' . $layout );
		}
		echo '</div>';
	}


	$output = ob_get_clean();

	$wp_query = $temp_query;
	wp_reset_postdata();

	return $output;


}



// 1.14 Sidebar
/*
Allows you to insert a dynamic sidebar anywhere
*/
add_shortcode( 'sidebar', COMPANYPREFIX . '_shortcode_sidebar' );
function bsh_shortcode_sidebar( $args ) {
	global $shortcode_style, $post, $wp_query;
	$defaults = array(
		'name'              => 'Sidebar',
	);
	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	ob_start();
	dynamic_sidebar( $name );
	$output = ob_get_clean();
	return $output;

}



/***********************************************/
/*         Shortcode Helper Functions          */
/***********************************************/

// 2.1 Shortcode Style Generator
function bsh_shortcode_style( $property, $value, $type = 'default' ) {
	global $shortcode_style;
	if( !empty( $value ) ) {
		if( $type === 'color' OR $type === 'gradient' OR $type === 'border-darken' ) {
			$value = bsh_determine_color( $value );
		}

		if( $type === 'gradient' ){
			$color = new Color( $value );
			$gradient = $color->makeGradient( 10 );
			$shortcode_style .= "
				background: #" . $gradient['light'] . ";
				background: -moz-linear-gradient(top,  #" . $gradient['light'] . " 0%, #" . $gradient['dark'] . " 100%);
				background: -webkit-gradient(linear, left top, left bottom, color-stop(0%,#" . $gradient['light'] . "), color-stop(100%,#" . $gradient['dark'] . "));
				background: -webkit-linear-gradient(top,  #" . $gradient['light'] . " 0%,#" . $gradient['dark'] . " 100%);
				background: -o-linear-gradient(top,  #" . $gradient['light'] . " 0%,#" . $gradient['dark'] . " 100%);
				background: -ms-linear-gradient(top,  #" . $gradient['light'] . " 0%,#" . $gradient['dark'] . " 100%);
				background: linear-gradient(to bottom,  #" . $gradient['light'] . " 0%,#" . $gradient['dark'] . " 100%);
				filter: progid:DXImageTransform.Microsoft.gradient( startColorstr='" . $gradient['light'] . "', endColorstr='" . $gradient['dark'] . "',GradientType=0 );
			";

		}
		elseif( $type === 'border-darken' ) {
			$color = new Color( $value );
			$shortcode_style .= $property . ': #' . $color->darken(10) . ';';
		}
		elseif( $type === 'radius' ) {
			$shortcode_style .= '
				-moz-border-radius: ' . $value . ';
				-khtml-border-radius: ' . $value . ';
				-ie-border-radius: ' . $value . ';
				-o-border-radius: ' . $value . ';
				border-radius: ' . $value . ';
			';
		}
		else {
			$css = $property . ': ' . $value;
			$shortcode_style .= $css . ';';
		}
	}
}

?>