<?php

/*
WPOnlineBackup_BootStrap - Workhouse for the overall backup
Coordinates the different types of backups: Files, Database tables.
*/

// Define the backup status codes
define( 'WPONLINEBACKUP_STATUS_NONE',		0 ); // Referenced manually in wponlinebackup.php Activate() as bootstrap not included
define( 'WPONLINEBACKUP_STATUS_STARTING',	1 );
define( 'WPONLINEBACKUP_STATUS_RUNNING',	2 );
define( 'WPONLINEBACKUP_STATUS_TICKING',	3 );

// Define the activity types
define( 'WPONLINEBACKUP_ACTIVITY_BACKUP',	0 );
define( 'WPONLINEBACKUP_ACTIVITY_AUTO_BACKUP',	1 );
define( 'WPONLINEBACKUP_ACTIVITY_RESTORE',	2 );

// Define the activity media types
define( 'WPONLINEBACKUP_MEDIA_UNKNOWN',		0 ); // Mainly for backwards compatibility when we didn't store the target for backups
define( 'WPONLINEBACKUP_MEDIA_DOWNLOAD',	1 );
define( 'WPONLINEBACKUP_MEDIA_EMAIL',		2 );
define( 'WPONLINEBACKUP_MEDIA_ONLINE',		3 );

// Define the activity completion status codes
define( 'WPONLINEBACKUP_COMP_RUNNING',		0 ); // Running
define( 'WPONLINEBACKUP_COMP_SUCCESSFUL',	1 ); // Successful
define( 'WPONLINEBACKUP_COMP_PARTIAL',		2 ); // Completed, but with errors (so SOME data was backed up)
define( 'WPONLINEBACKUP_COMP_UNEXPECTED',	3 ); // Failed - timed out and never recovered - WP-Cron broken?
define( 'WPONLINEBACKUP_COMP_FAILED',		4 ); // Failed - mainly where backup file could not be opened, or online transmission fails for incrementals
define( 'WPONLINEBACKUP_COMP_TIMEOUT',		5 ); // Failed - timed out too many times and never made progress (reached max_frozen_retries)
define( 'WPONLINEBACKUP_COMP_SLOWTIMEOUT',	6 ); // Failed - timed out too many times and did make progress each time (reaches max_progress_retries)

// Define the event codes
define( 'WPONLINEBACKUP_EVENT_INFORMATION',	0 );
define( 'WPONLINEBACKUP_EVENT_WARNING',		1 );
define( 'WPONLINEBACKUP_EVENT_ERROR',		2 );

// Define the bin codes and names
define( 'WPONLINEBACKUP_BIN_DATABASE',		1 );
define( 'WPONLINEBACKUP_BIN_FILESYSTEM',	2 );

// WP-Cron runs events in succession - so we could have Perform_Check() running, and then immediately after
// (in the same WP-Cron process) Perform(). So we use this global to ensure we run once per process!
// We also use this to detect how long we've been running and adjust max_execution_time as necessary
$WPOnlineBackup_Init = time();
$WPOnlineBackup_Perform_Once = false;

class WPOnlineBackup_BootStrap
{
	/*private*/ var $WPOnlineBackup;

	/*private*/ var $status = null;
	/*private*/ var $last_tick_status = null;
	/*private*/ var $activity_id;
	/*private*/ var $start_time;

	/*private*/ var $processors = array();
	/*private*/ var $stream = null;

// Cache
	/*private*/ var $max_execution_time;
	/*private*/ var $recovery_time;

	/*public*/ function WPOnlineBackup_BootStrap( & $WPOnlineBackup )
	{
		global $wpdb;

		$this->WPOnlineBackup = & $WPOnlineBackup;

		$this->status = array();

// Grab the data from the database
		list ( $this->status['status'], $this->status['time'], $this->status['counter'], $this->status['progress'] ) =
			$wpdb->get_row(
				'SELECT status, time, counter, progress FROM `' . $wpdb->prefix . 'wponlinebackup_status` LIMIT 1',
				ARRAY_N
			);

		$this->status['progress'] = @unserialize( $this->status['progress'] );

// If progress data is invalid, blank it out, otherwise, grab the activity_id
		if ( $this->status['progress'] === false ) $this->status['progress'] = array( 'activity_id' => null );
		else $this->activity_id = $this->status['progress']['activity_id'];
	}

	/*public*/ function Fetch_Status()
	{
		return $this->status;
	}

	/*private*/ function Update_Status( $new_status = false, $new_counter = false )
	{
		global $wpdb;

// If we didn't give a status, leave it the same
		if ( $new_status === false ) $new_status = $this->status['status'];

// Increase the progress counter so we don't fail to update
		if ( $new_counter === false ) $new_counter = $this->status['counter'] + 1;

// Serialize and escape_by_ref (uses _real_escape - better)
		$q_new_progress = serialize( $this->status['progress'] );
		$wpdb->escape_by_ref( $q_new_progress );

// Update the database
		$now = time();
		$result = $wpdb->query(
			'UPDATE `' . $wpdb->prefix . 'wponlinebackup_status` ' .
			'SET status = ' . $new_status . ', ' .
				'time = ' . $now . ', ' .
				'counter = ' . $new_counter . ', ' .
				'progress = \'' . $q_new_progress . '\' ' .
			'WHERE status = ' . $this->status['status'] . ' ' .
				'AND counter = ' . $this->status['counter'] . ' ' .
				'AND time = ' . $this->status['time']
		);

		if ( $result ) {

// We updated the row, store the time
			$this->status['status'] = $new_status;
			$this->status['time'] = $now;
			$this->status['counter'] = $new_counter;

// Continue
			return true;

		}

// No row was updated, the mutex lock is lost - abort
		return false;
	}

	/*public*/ function Start( $config, $type, $with_immediate_effect = false )
	{
// Check to see if a backup is already running, and grab the backup lock if possible
// If a backup is running, but the time_presumed_dead period has passed, we presume the backup to have failed, and allow another to be started
		if (
			(
					$this->status['status'] != WPONLINEBACKUP_STATUS_NONE
				&&	$this->status['time'] > time() - $this->WPOnlineBackup->Get_Setting( 'time_presumed_dead' )
			)
		) {
			return false;
		}

		$this->WPOnlineBackup->Load_Settings();

// Check we have a .htaccess and attempt to copy one in
		if ( !@file_exists( $this->WPOnlineBackup->Get_Setting( 'local_tmp_dir' ) . '/.htaccess' ) )
			@copy( WPONLINEBACKUP_PATH . '/tmp.httpd', $this->WPOnlineBackup->Get_Setting( 'local_tmp_dir' ) . '/.htaccess' );

// Populate a new activity, if this fails then there is a problem with the internal database tables - needs reactivation - return the message
		$start = time();

		if ( ( $ret = $this->Start_Activity( $config, $type, $start ) ) !== true ) return $ret;

		if ( ( $ret = $this->Log_Event( WPONLINEBACKUP_EVENT_INFORMATION, __( 'Backup starting...' , 'wponlinebackup') ) ) !== true ) {
			$this->End_Activity( WPONLINEBACKUP_COMP_FAILED );
			return $ret;
		}

// Prepare the progress and its tracker
		$this->status['progress'] = array(
			'start_time'		=> $start,			// The start time of the backup
			'initialise'		=> 1,				// Whether or not initialisation is complete
			'activity_id'		=> $this->activity_id,		// Activity ID this backup represents
			'message'		=> __( 'Waiting for the backup to start...' , 'wponlinebackup'),	// Message to show in monitoring page
			'config'		=> $config,			// Backup configuration
			'frozen_timeouts'	=> 0,				// Timeouts with no progress (compared with max_frozen_timeouts)
			'last_timeout'		=> null,			// Progress at last timeout (used to detect a frozen timeout)
			'progress_timeouts'	=> 0,				// Timeouts with progress (compared with max_progress_timeouts)
			'errors'		=> 0,				// Number of errors
			'warnings'		=> 0,				// Number of warnings
			'jobs'			=> array(),			// Job list - the backup job works its way through these - we populate this below
			'jobcount'		=> 0,				// Number of jobs - used to calculate progress in percent
			'jobdone'		=> 0,				// Number of jobs done
			'rotation'		=> 0,				// Do we need to rotate due to failure?
			'file'			=> null,			// The backup file - we populate this below
			'file_set'		=> null,			// The resulting backup files
			'rcount'		=> 0,				// Total number of files approached (not necessarily stored)
			'rsize'			=> 0,				// Total size of files approached (not necessarily stored)
			'ticks'			=> 0,				// Tick count
			'update_ticks'		=> $this->WPOnlineBackup->Get_Setting( 'update_ticks' ), // Number of ticks before update. We decrease to 1 on timeout.
			'revert_update_ticks'	=> 0,				// When update_ticks is set to 1 we use this to decide when to change it back
			'tick_progress'		=> array( 0 => false, 1 => 0 ),	// Tick progress when update_ticks is 1 and we're taking care
			'performs'		=> 0,				// Perform count
			'nonce'			=> '',				// Nonce for online collection if we need it
			'bsn'			=> 0,				// Keep track of BSN for incremental backups
			'cache'			=> array(),			// Cached settings - we clear them after backup
		);

		$this->status['progress']['cache']['max_log_age'] = $this->WPOnlineBackup->Get_Setting( 'max_log_age' );

		if ( $config['target'] == 'online' ) {

			$this->status['progress']['cache']['username'] = $this->WPOnlineBackup->Get_Setting( 'username' );
			$this->status['progress']['cache']['password'] = $this->WPOnlineBackup->Get_Setting( 'password' );

			if ( $this->status['progress']['cache']['username'] == '' ) {

				$this->Log_Event(
					WPONLINEBACKUP_EVENT_ERROR,
					$ret = __( 'The backup could not be started; an online backup cannot be performed if the plugin is not currently logged into the online backup servers. Please click \'Online Backup Settings\' and login to enable online backup.' , 'wponlinebackup')
				);
				$this->End_Activity( WPONLINEBACKUP_COMP_FAILED );
				return $ret;

			}

		}

		$this->status['progress']['cache']['enc_type'] = $this->WPOnlineBackup->Get_Setting( 'encryption_type' );
		$this->status['progress']['cache']['enc_key'] = $this->WPOnlineBackup->Get_Setting( 'encryption_key' );

		if ( !$this->Update_Status( WPONLINEBACKUP_STATUS_STARTING, 0 ) ) return false;

// Schedule the backup check thread for 65 seconds in the future
		wp_schedule_single_event( time() + 65, 'wponlinebackup_perform_check' );

		if ( $with_immediate_effect ) {

// A scheduled backup so we start with immediate effect
			$this->Perform( true );

		} else {

// Manual - Schedule the backup thread for in 5 seconds - hopefully after this page load so we can show the progress from the start if manually starting
			wp_schedule_single_event( time() + 5, 'wponlinebackup_perform' );

		}

// Backup has started and is ready to run!
		return true;
	}

	/*public*/ function Start_Activity( $config, $type, $start )
	{
		global $wpdb;

		// Cleanup any old stale activity entries - any that are LESS than the current time and have NULL completion time - care not for the result
		$wpdb->query(
			'UPDATE `' . $wpdb->prefix . 'wponlinebackup_activity_log` ' .
			'SET end = ' . $start . ', ' .
				'comp = ' . WPONLINEBACKUP_COMP_UNEXPECTED . ' ' .
			'WHERE end IS NULL'
		);

		// Resolve the media
		switch ( $config['target'] ) {
			case 'download':
				$media = WPONLINEBACKUP_MEDIA_DOWNLOAD;
				break;
			case 'email':
				$media = WPONLINEBACKUP_MEDIA_EMAIL;
				break;
			case 'online':
				$media = WPONLINEBACKUP_MEDIA_ONLINE;
				break;
			default:
				$media = WPONLINEBACKUP_MEDIA_UNKNOWN;
				break;
		}

		// Insert a new activity row. Return false if we fail
		if ( $wpdb->query(
			'INSERT INTO `' . $wpdb->prefix . 'wponlinebackup_activity_log` ' .
			'(start, end, type, comp, media, compressed, encrypted, errors, warnings, bsize, bcount, rsize, rcount) ' .
			'VALUES ' .
			'(' .
				$start . ', ' .				// Start time
				'NULL, ' .				// End time is null as the activity has yet to finish
				$type . ', ' .				// Activity type
				WPONLINEBACKUP_COMP_RUNNING . ', ' .	// Current status is running
				$media . ', ' .				// Media
				'0, ' .					// Compressed?
				'0, ' .					// Encrypted?
				'0, ' .					// Number of errors - start at 0
				'0, ' .					// Number of warnings - start at 0
				'0, ' .					// These four fields are described in wponlinebackup.php during creation
				'0, ' .					// -
				'0, ' .					// -
				'0' .					// -
			')'
		) === false ) return WPOnlineBackup::Get_WPDB_Last_Error();

		// Store the activity_id
		$this->activity_id = $wpdb->insert_id;

		return true;
	}

	/*public*/ function End_Activity( $status, $progress = false )
	{
		global $wpdb;

		if ( $progress === false ) $progress = array(
			'file_set'	=> array(
				'compressed'	=> 0,
				'encrypted'	=> 0,
				'size'		=> 0,
				'files'		=> 0,
			),
			'rsize'		=> 0,
			'rcount'	=> 0,
			'errors'	=> 0,
			'warnings'	=> 0,
		);

// Update the loaded activity
// - care not for the return status, best to kick off errors during starting a backup, then starting a backup AND finishing a backup
//   that and we could be finishing the backup due to database errors anyways - so reporting here would be completely redundant
		$wpdb->update(
			$wpdb->prefix . 'wponlinebackup_activity_log',
			array(
				'end'		=> time(),	// Set end time to current time
				'comp'		=> $status,	// Set completion status to the given status
				'errors'	=> $progress['errors'],
				'warnings'	=> $progress['warnings'],
				'compressed'	=> $progress['file_set']['compressed'],
				'encrypted'	=> $progress['file_set']['encrypted'],
				'bsize'		=> $progress['file_set']['size'],
				'bcount'	=> $progress['file_set']['files'],
				'rsize'		=> $progress['rsize'],
				'rcount'	=> $progress['rcount'],
			),
			array(
				'activity_id'	=> $this->activity_id,
			),
			'%d',
			'%d'
		);
	}

	/*public*/ function Log_Event( $type, $event )
	{
		global $wpdb;

// Increase error count if an error is being logged
		if ( $type == WPONLINEBACKUP_EVENT_ERROR )
			$this->status['progress']['errors']++;
		else if ( $type == WPONLINEBACKUP_EVENT_WARNING )
			$this->status['progress']['warnings']++;

// Insert the event
		$res = $wpdb->insert(
			$wpdb->prefix . 'wponlinebackup_event_log',
			array(
				'activity_id'	=> $this->activity_id,	// Current activity
				'time'		=> time(),		// Set event time to current time
				'type'		=> $type,		// Set event type to given type
				'event'		=> $event,		// Set event message to given message
			)
		);

		if ( $res === false ) return WPOnlineBackup::Get_WPDB_Last_Error();

		return true;
	}

	/*public*/ function DBError( $file, $line, $friendly = false )
	{
		$this->Log_Event(
			WPONLINEBACKUP_EVENT_ERROR,
			__( 'A database operation failed.' , 'wponlinebackup') . PHP_EOL .
				__( 'Please try reinstalling the plugin - in most cases this will repair the database.' , 'wponlinebackup') . PHP_EOL .
				__( 'Please contact support if the issue persists, providing the complete event log for the activity. Diagnostic information follows:' , 'wponlinebackup') . PHP_EOL . PHP_EOL .
				'Failed at: ' . $file . '(' . $line . ')' . PHP_EOL .
				WPOnlineBackup::Get_WPDB_Last_Error()
		);

		if ( $friendly === false )
			$friendly = __( 'A database operation failed.' , 'wponlinebackup');

		return $friendly;
	}

	/*public*/ function FSError( $file, $line, $of, $ret, $friendly = false )
	{
		$this->Log_Event(
			WPONLINEBACKUP_EVENT_ERROR,
			( $of === false ? __( 'A filesystem operation failed.' , 'wponlinebackup') : sprintf( __( 'A filesystem operation failed while processing %s for backup.' , 'wponlinebackup'), $of ) ) . PHP_EOL .
				__( 'If the following error message is not clear as to the problem and the issue persists, please contact support providing the complete event log for the activity. Diagnostic information follows:' , 'wponlinebackup') . PHP_EOL . PHP_EOL .
				'Failed at: ' . $file . '(' . $line . ')' . PHP_EOL .
				$ret
		);

		if ( $friendly === false )
			$friendly = __( 'A filesystem operation failed.' , 'wponlinebackup');

		return $friendly;
	}

	/*public*/ function COMError( $file, $line, $ret, $friendly = false )
	{
		$this->Log_Event(
			WPONLINEBACKUP_EVENT_ERROR,
			__( 'A transmission operation failed.' , 'wponlinebackup') . PHP_EOL .
				__( 'If the following error message is not clear as to the problem and the issue persists, please contact support providing the complete event log for the activity. Diagnostic information follows:' , 'wponlinebackup') . PHP_EOL . PHP_EOL .
				'Failed at: ' . $file . '(' . $line . ')' . PHP_EOL .
				$ret
		);

		if ( $friendly === false )
			$friendly = __( 'Communication with the online vault failed.' , 'wponlinebackup');

		return $friendly;
	}

	/*public*/ function MALError( $file, $line, $xml, $parser_ret = false )
	{
		$this->Log_Event(
			WPONLINEBACKUP_EVENT_ERROR,
			__( 'An online request succeeded but was malformed.' , 'wponlinebackup') . PHP_EOL .
				__( 'Please contact support if the issue persists, providing the complete event log for the activity. Diagnostic information follows:' , 'wponlinebackup') . PHP_EOL . PHP_EOL .
				'Failed at: ' . $file . '(' . $line . ')' . PHP_EOL .
				( $parser_ret === false ? 'XML parser succeeded' : 'XML parser: ' . $parser_ret . PHP_EOL ) .
				'XML log:' . PHP_EOL . $xml->log
		);

		return __( 'Communication with the online vault failed.' , 'wponlinebackup');
	}

	/*public*/ function Register_Temp( $temp )
	{
		$temps = get_option( 'wponlinebackup_temps', array() );
		
		$temps[] = $temp;
		update_option( 'wponlinebackup_temps', $temps );
	}

	/*public*/ function Unregister_Temp( $temp )
	{
		$temps = get_option( 'wponlinebackup_temps', array() );
		
		if ( ( $key = array_search( $temp, $temps ) ) !== false ) {

			unset( $temps[$key] );
			update_option( 'wponlinebackup_temps', $temps );

		}
	}

	/*public*/ function Clean_Temps()
	{
		$temps = get_option( 'wpnlinebackup_temps', array() );

		foreach ( $temps as $item )
			@unlink( $item );
	}

	/*public*/ function On_Shutdown()
	{
		if ( $this->status['status'] == WPONLINEBACKUP_STATUS_RUNNING ) {

			// Try to recover status
			if ( !is_null( $this->last_tick_status ) ) {

				// Copy last tick status to current status
				$this->status = $this->last_tick_status;

				// Update status, leave if we've lost the lock
				if ( !$this->Update_Status( WPONLINEBACKUP_STATUS_TICKING ) ) exit;

			}

		} else if ( $this->status['status'] != WPONLINEBACKUP_STATUS_TICKING ) {

			// Not running, not ticking, exit
			exit;

		}

		// Just in case the below fails, schedule next event
		wp_schedule_single_event( time() + 5, 'wponlinebackup_perform' );

		// Attempt to kick start the backup again - this bit based on spawn_cron()
		$do_url = site_url( '/' ) . '?wponlinebackup_do';
		wp_remote_post(
			$do_url,
			array(
				'timeout' => 0.01,
				'blocking' => false,
				'sslverify' => apply_filters( 'https_local_ssl_verify', true ),
			)
		);
	}

	/*public*/ function Tick( $next = false, $update = false )
	{
		if ( time() - $this->start_time > $this->recovery_time ) {

// We've run for way too long - Perform_Check will have run by now, so just quit
			$exit = true;

		} else {

			$exit = false;

			if ( $next || ( $run_time = time() - $this->start_time ) > $this->max_execution_time ) {

				// If we're forcing next, ensure we've run for at least 5 seconds
				if ( $next && $run_time < $this->max_execution_time ) {

					// Sleep a bit, but not too long as to reach max_execution_time and 5 seconds at most
					// We do this to prevent eating too much resources on the server
					sleep( min( 5, $this->max_execution_time - 2 - $run_time ) );

				}

				$this->status['progress']['rotation']--;

// Update the stream state
				if ( is_object( $this->stream ) ) $this->status['progress']['file']['state'] = $this->stream->Save();
				else $this->status['progress']['file'] = null;

// Reset the update ticks and tick count
				$this->status['progress']['ticks'] = 0;
				$this->status['progress']['update_ticks'] = $this->WPOnlineBackup->Get_Setting( 'update_ticks' );

				// Update status
				$this->Update_Status( WPONLINEBACKUP_STATUS_TICKING );

				$exit = true;

			} else {

				if ( $this->status['progress']['update_ticks'] == 1 ) {

// We made progress, so clear the frozen timeouts counter
					$this->status['progress']['frozen_timeouts'] = 0;

// We'll store a 0 tick count... but we'll keep the actual tick count so we know when to revert update_ticks
					$ticks = $this->status['progress']['ticks'];
					$this->status['progress']['ticks'] = 0;

					$update = true;

// We're taking our time at the moment and always updating; if we hit the revert update_ticks value we can revert it
					if ( ++$this->status['progress']['ticks'] >= $this->status['progress']['revert_update_ticks'] ) {

						$this->status['progress']['update_ticks'] = $this->WPOnlineBackup->Get_Setting( 'update_ticks' );

					}

				} else {

// Only update if tick count reached - speeds things up alot
					if ( $update || ++$this->status['progress']['ticks'] >= $this->status['progress']['update_ticks'] ) {

						$this->status['progress']['ticks'] = 0;

						$update = true;

					}

				}

				// Update the stream state
				if ( is_object( $this->stream ) ) $this->status['progress']['file']['state'] = $this->stream->Save();
				else $this->status['progress']['file'] = null;

				if ( $update ) {

					// Update status, leave if we've lost the lock
					if ( !$this->Update_Status() ) $exit = true;

				} else {

					$this->last_tick_status = $this->status;

				}

				if ( $this->status['progress']['update_ticks'] == 1 ) {

// Put the tick count back...
					$this->status['progress']['ticks'] = $ticks;

				}

			}

		}

		if ( $exit ) {

			$this->CleanUp_Processors( true );

			exit;

		}

		return true;
	}

	/*public*/ function Perform_Check()
	{
		if ( $this->status['status'] == WPONLINEBACKUP_STATUS_NONE ) exit;

		$this->WPOnlineBackup->Load_Settings();

		if (
				$this->status['status'] != WPONLINEBACKUP_STATUS_NONE
			&&	$this->status['time'] <= time() - $this->WPOnlineBackup->Get_Setting( 'time_presumed_dead' )
		) return;

		wp_clear_scheduled_hook( 'wponlinebackup_perform_check' );

// Ticking over or starting? Boot it off now
		if ( $this->status['status'] == WPONLINEBACKUP_STATUS_TICKING || $this->status['status'] == WPONLINEBACKUP_STATUS_STARTING ) {

// Schedule again in future in 60 seconds
			wp_schedule_single_event( time() + 65, 'wponlinebackup_perform_check' );

			$this->Perform( true );

			return;

		}

		if ( $this->status['time'] > time() - $this->WPOnlineBackup->Get_Setting( 'timeout_recovery_time' ) ) {

// Schedule again in future in 60 seconds
			wp_schedule_single_event( time() + 65, 'wponlinebackup_perform_check' );
// Timeout didn't occur, just exit
			return;

		}

		$this->activity_id = $this->status['progress']['activity_id'];

		$last_timeout = md5( serialize( $this->status['progress']['jobs'] ) );

// Did we make progress?
		if ( !is_null( $this->status['progress']['last_timeout'] ) ) {

			if ( $last_timeout == $this->status['progress']['last_timeout'] ) {

				if ( ++$this->status['progress']['frozen_timeouts'] > $this->WPOnlineBackup->Get_Setting( 'max_frozen_retries' ) ) {

					// Remove any schedule
					wp_clear_scheduled_hook( 'wponlinebackup_perform' );

					// Timeout occurred
					$this->Log_Event(
						WPONLINEBACKUP_EVENT_WARNING,
						$ret = __( 'The backup timed out too many times and no progress was made on any attempt. Your server may be running extremely slow at this time - try scheduling the backup during a quieter period.' , 'wponlinebackup')
					);

					$this->End_Activity( WPONLINEBACKUP_COMP_TIMEOUT );

					$this->status['progress']['message'] = $ret;
					unset( $this->status['progress']['cache'] );

					$this->Update_Status( WPONLINEBACKUP_STATUS_NONE );

					return;

				}

			}

		}

// Reset tick count to 1 so we constantly update to try get past this blockage that caused the timeout, also store the revert count
		$this->status['progress']['revert_update_ticks'] = $this->status['progress']['update_ticks'];
		$this->status['progress']['update_ticks'] = 1;

		$this->status['progress']['last_timeout'] = $last_timeout;

		if ( ++$this->status['progress']['progress_timeouts'] > ( $max_progress_retries = $this->WPOnlineBackup->Get_Setting( 'max_progress_retries' ) ) && $max_progress_retries != 0 ) {

			// Remove any schedule
			wp_clear_scheduled_hook( 'wponlinebackup_perform' );

			// Timeout occurred
			$this->Log_Event(
				WPONLINEBACKUP_EVENT_WARNING,
				$ret = __( 'The backup timed out too many times and was progressing too slowly. Your server may be running extremely slow at this time - try scheduling the backup during a quieter period.' , 'wponlinebackup')
			);

			$this->End_Activity( WPONLINEBACKUP_COMP_SLOWTIMEOUT );

			$this->status['progress']['message'] = $ret;
			unset( $this->status['progress']['cache'] );

			$this->Update_Status( WPONLINEBACKUP_STATUS_NONE );

			return;

		}

// Schedule again in future in 60 seconds
		wp_schedule_single_event( time() + 65, 'wponlinebackup_perform_check' );

// Update the message - bit vague but we don't really know if this is going to be a timeout issue or not
		$this->status['progress']['message'] = __( 'A timeout occurred during backup (large files and slow servers are the common cause of this); trying again...' , 'wponlinebackup');

// Run the backup now
		$this->Perform( true );
	}

	/*public*/ function Perform( $ignore_timeout = false )
	{
		global $WPOnlineBackup_Perform_Once, $WPOnlineBackup_Init;

// Check we haven't already run once during this PHP session
		if ( $WPOnlineBackup_Perform_Once === true ) return;
		$WPOnlineBackup_Perform_Once = true;

		// Register shutdown event
		register_shutdown_function( array( $this, 'On_Shutdown' ) );

		// Remove any schedule
		wp_clear_scheduled_hook( 'wponlinebackup_perform' );

		if ( !$ignore_timeout ) {

			if ( $this->status['status'] != WPONLINEBACKUP_STATUS_TICKING && $this->status['status'] != WPONLINEBACKUP_STATUS_STARTING ) exit;

			$this->WPOnlineBackup->Load_Settings();

			if (
					$this->status['status'] != WPONLINEBACKUP_STATUS_NONE
				&&	$this->status['time'] <= time() - $this->WPOnlineBackup->Get_Setting( 'time_presumed_dead' )
			) return;

			$this->activity_id = $this->status['progress']['activity_id'];

			if ( $this->status['time'] <= time() - $this->WPOnlineBackup->Get_Setting( 'timeout_recovery_time' ) ) return;

		}

		$this->start_time = time();

// Test safe mode
		$safe_mode = ini_get( 'safe_mode' );
		if ( !is_bool( $safe_mode ) ) $safe_mode = preg_match( '/^on$/i', $safe_mode );

		if ( $safe_mode ) {

// Cannot change time limit in safe mode, so offset the max_execution_time based on how much time we've lost since initialisation, but give a minimum of 5 seconds
			$offset = time() - $WPOnlineBackup_Init;
			$this->max_execution_time = ( $offset > $this->WPOnlineBackup->Get_Setting( 'max_execution_time' ) ? false : $this->WPOnlineBackup->Get_Setting( 'max_execution_time' ) - $offset );
			if ( $this->max_execution_time === false ) $this->max_execution_time = min( 5, $this->WPOnlineBackup->Get_Setting( 'max_execution_time' ) );

		} else {

			$this->max_execution_time = $this->WPOnlineBackup->Get_Setting( 'max_execution_time' );

// Just set new time limit - don't be over zealous as we don't want to cause issues on the server, so set to twice the max time
// We should normally pause and resume after the max time, but if we do hang, this will give a little leeway which can only be good
			set_time_limit( $this->max_execution_time * 2 );

		}

		$this->recovery_time = $this->WPOnlineBackup->Get_Setting( 'timeout_recovery_time' );

		$this->status['progress']['performs']++;

// Clear old temporary files if possible
		$this->Clean_Temps();

// Store the previous rotation value - we use this when recreating the stream
		$rotation = $this->status['progress']['rotation'];

// Increase the rotation so if we get interrupted, we implicitly rotate. We'll decrease this back if we exit gracefully.
		$this->status['progress']['rotation']++;

// Check we still have the lock
		if ( !$this->Update_Status( WPONLINEBACKUP_STATUS_RUNNING ) ) exit;

// Initialise if required
		if ( $this->status['progress']['initialise'] && ( $ret = $this->Initialise() ) !== true ) {

			$this->Log_Event(
				WPONLINEBACKUP_EVENT_ERROR,
				$ret = sprintf( __( 'The backup failed to initialise: %s.' , 'wponlinebackup'), $ret )
			);

			$this->End_Activity( WPONLINEBACKUP_COMP_FAILED );

			$this->status['progress']['message'] = $ret;
			unset( $this->status['progress']['cache'] );

			$this->Update_Status( WPONLINEBACKUP_STATUS_NONE );

			return;

		}

		if ( is_null( $this->stream ) && !is_null( $this->status['progress']['file'] ) ) {

			require_once WPONLINEBACKUP_PATH . '/include/' . strtolower( $this->status['progress']['file']['type'] ) . '.php';

			$name = 'WPOnlineBackup_' . $this->status['progress']['file']['type'];
			$this->stream = new $name( $this->WPOnlineBackup );
			if ( ( $ret = $this->stream->Load( $this->status['progress']['file']['state'], $rotation ) ) !== true )
				$this->stream = null;

// No longer need state information - we can scrap it. When we save we will generate a new state.
// This also frees a significant amount of memory.
			unset( $this->status['progress']['file']['state'] );

		} else {

			$ret = true;

		}

		if ( $ret === true ) {

			$ret = $this->Backup();

		}

		// Cleanup
		$this->CleanUp_Processors();

		if ( $ret !== true ) {

			if ( !is_null( $this->stream ) ) $this->stream->CleanUp();
			else if ( !is_null( $this->status['progress']['file_set'] ) ) {
				if ( !is_array( $this->status['progress']['file_set']['file'] ) ) @unlink( $this->status['progress']['file_set']['file'] );
				else foreach ( $this->status['progress']['file_set']['file'] as $file ) @unlink( $file );
			}

			$this->Log_Event(
				WPONLINEBACKUP_EVENT_ERROR,
				$ret = sprintf( __( 'The backup failed: %s' , 'wponlinebackup'), $ret )
			);

			$this->End_Activity( WPONLINEBACKUP_COMP_FAILED );

			$this->status['progress']['message'] = $ret;
			unset( $this->status['progress']['cache'] );

			$this->Update_Status( WPONLINEBACKUP_STATUS_NONE );

			return;

		}

		if ( $this->status['progress']['config']['target'] == 'online' ) {

// Update the backup serial number
			update_option( 'wponlinebackup_bsn', $this->status['progress']['bsn'] );

// Cleanup the files, we don't keep them for online backups
			foreach ( $this->status['progress']['file_set']['file'] as $file ) @unlink( $file );
			$this->status['progress']['file_set']['size'] = array_sum( $this->status['progress']['file_set']['size'] );

		} else if ( $this->status['progress']['config']['target'] == 'email' ) {

// We emailed the backup - no longer need to keep it
			@unlink( $this->status['progress']['file_set']['file'] );

		} else {

// Store the file path, we keep a full backup until deleted manually
			update_option( 'wponlinebackup_last_full', $this->status['progress']['file_set'] );

		}

		$this->Log_Event(
			WPONLINEBACKUP_EVENT_INFORMATION,
			__( 'Backup complete.' , 'wponlinebackup')
		);

		$this->End_Activity(
			$this->status['progress']['errors'] ? WPONLINEBACKUP_COMP_PARTIAL : WPONLINEBACKUP_COMP_SUCCESSFUL,
			$this->status['progress']
		);

		unset( $this->status['progress']['cache'] );
		$this->Update_Status( WPONLINEBACKUP_STATUS_NONE );

		wp_clear_scheduled_hook( 'wponlinebackup_perform' );
		wp_clear_scheduled_hook( 'wponlinebackup_perform_check' );
	}

	/*private*/ function Initialise()
	{
		global $wpdb;

		$progress = & $this->status['progress'];

// First of all, clear back activity logs
		if ( $progress['initialise'] < 2 ) {

			do {

				if ( ( $ret = $wpdb->query(
					'DELETE a, e FROM `' . $wpdb->prefix . 'wponlinebackup_activity_log` a ' .
						'LEFT JOIN `' . $wpdb->prefix . 'wponlinebackup_event_log` e ON (e.activity_id = a.activity_id) ' .
					'WHERE a.start < ' . strtotime( '-' . $progress['cache']['max_log_age'] . ' months', $progress['start_time'] )
				) ) === false ) return $this->DBError( __LINE__, __FILE__ );

				if ( !$ret ) $progress['initialise'] = 2;

				$this->Tick();

			} while ( $ret );

		}

		if ( $progress['initialise'] < 3 ) {

			if ( $progress['config']['target'] != 'online' ) {

				$last_full = get_option( 'wponlinebackup_last_full', array() );

				if ( array_key_exists( 'file', $last_full ) ) {

					$progress['message'] = __( 'Deleting previous Full Backup...' , 'wponlinebackup');

					@unlink( $last_full['file'] );

					$this->Log_Event(
						WPONLINEBACKUP_EVENT_INFORMATION,
						__( 'Previous Full Backup deleted.' , 'wponlinebackup')
					);

				}

				update_option( 'wponlinebackup_last_full', array() );

			}

			$progress['message'] = __( 'Initialising...' , 'wponlinebackup');

			if ( $progress['config']['target'] == 'online' ) {

				$progress['jobs'][] = array(
					'processor'		=> 'transmission',
					'progress'		=> 0,
					'progresslen'		=> 5,
					'action'		=> 'synchronise',
					'total_items'		=> 0,
					'total_generations'	=> 0,
					'done_items'		=> 0,
					'done_generations'	=> 0,
				);

			}

			$progress['cleanups'] = array();

			$progress['initialise'] = 3;

			$this->Tick();

		}

		if ( $progress['initialise'] < 4 ) {

// Are we backing up the database?
			if ( $progress['config']['backup_database'] ) {

				require_once WPONLINEBACKUP_PATH . '/include/tables.php';
				$tables = new WPOnlineBackup_Backup_Tables( $this->WPOnlineBackup );

// Initialise - pass ourself so we can log events, and also pass the progress and its tracker
				if ( ( $ret = $tables->Initialise( $this, $progress ) ) !== true ) return $ret;

			}

			$progress['initialise'] = 4;

			$this->Tick();

		}

		if ( $progress['initialise'] < 5 ) {

// Are we backing up the filesystem?
			if ( $progress['config']['backup_filesystem'] ) {

				require_once WPONLINEBACKUP_PATH . '/include/files.php';
				$files = new WPOnlineBackup_Backup_Files( $this->WPOnlineBackup );

// Initialise - pass ourselves so we can log events, and also pass the progress and its tracker
				if ( ( $ret = $files->Initialise( $this, $progress ) ) !== true ) return $ret;

			}

			$progress['initialise'] = 5;

			$this->Tick();

		}

		$progress['jobs'][] = array(
			'processor'	=> 'reconstruct',
			'progress'	=> 0,
			'progresslen'	=> 5,
		);

		if ( $progress['config']['target'] == 'online' ) {

			$progress['jobs'][] = array(
				'processor'		=> 'transmission',
				'progress'		=> 0,
				'progresslen'		=> 10,
				'action'		=> 'transmit',
				'total'			=> 0,
				'done'			=> 0,
				'done_retention'	=> 0,
				'retention_size'	=> 0,
				'new_bsn'		=> 0,
			);

		} else if ( $progress['config']['target'] == 'email' ) {

			$progress['jobs'][] = array(
				'processor'		=> 'email',
				'progress'		=> 0,
				'progresslen'		=> 10,
			);			

		}

		$progress['jobs'] = array_merge( $progress['jobs'], $progress['cleanups'] );

		unset( $progress['cleanups'] );

		foreach ( $progress['jobs'] as $job ) {

			$progress['jobcount'] += $job['progresslen'];

		}

// Prepare the stream configuration
// - the streams use this configuration instead of the central configuration so we can use different settings in different streams
		$config = array(
			'designated_path'	=> $this->WPOnlineBackup->Get_Setting( 'local_tmp_dir' ),
			'compression'		=> $this->WPOnlineBackup->Get_Env( 'deflate_available' ) ? 'DEFLATE' : 'store',
			'encryption'		=> $progress['cache']['enc_type'],
			'encryption_key'	=> $progress['cache']['enc_key'],
		);

		if ( !@file_exists( $config['designated_path'] ) ) @mkdir( $config['designated_path'], 0700 );

		if ( $progress['config']['target'] != 'online' ) {

// Full backups need storing separately
			$config['designated_path'] .= '/full';
			if ( !@file_exists( $config['designated_path'] ) )
				if ( @mkdir( $config['designated_path'], 0700 ) === false ) return OBFW_Exception();

		}

// Set up the required stream
		if ( $progress['config']['target'] == 'online' ) {

			$stream_type = 'Stream_Delta';

		} else {

			$stream_type = 'Stream_Full';

		}

		require_once WPONLINEBACKUP_PATH . '/include/' . strtolower( $stream_type ) . '.php';

		$name = 'WPOnlineBackup_' . $stream_type;
		$this->stream = new $name( $this->WPOnlineBackup );

// Open the file
		if ( ( $ret = $this->stream->Open( $config, html_entity_decode( get_bloginfo('name'), ENT_QUOTES, get_bloginfo('charset') ), html_entity_decode( get_bloginfo('description'), ENT_QUOTES, get_bloginfo('charset') ) ) ) !== true ) return $ret;

// Store the stream state so we can load it when performing
		$progress['file'] = array(
			'type'	=> $stream_type,
			'state'	=> $this->stream->Save(),
		);

		$progress['initialise'] = 0;

		$this->Tick();

		$this->Log_Event(
			WPONLINEBACKUP_EVENT_INFORMATION,
			__( 'Initialisation completed.' , 'wponlinebackup')
		);

// Success
		return true;
	}

	/*private*/ function CleanUp_Processors( $ticking = false )
	{
// For each processor we have loaded, clean it up
		foreach ( $this->processors as $processor ) {

			$processor->CleanUp( $ticking );

		}
	}

	/*private*/ function Fetch_Processor( $processor )
	{
// If we don't have the processor loaded already, load it
		if ( !array_key_exists( $processor, $this->processors ) ) {

			require_once WPONLINEBACKUP_PATH . '/include/' . $processor . '.php';

			$class = 'WPOnlineBackup_Backup_' . ucfirst( $processor );
			$this->processors[$processor] = new $class( $this->WPOnlineBackup );

		}

		return $this->processors[$processor];
	}

	/*private*/ function Backup()
	{
		@ignore_user_abort( true );

// Try to increase the time_limit to twice what we will actually run the script for - it will help make timeouts happen gracefully
// - but only if it isn't already higher, and if safe mode is not active
		if ( !ini_get( 'safe_mode' ) && ini_get( 'max_execution_time' ) < $this->WPOnlineBackup->Get_Setting( 'max_execution_time' ) * 2 ) @set_time_limit( $this->WPOnlineBackup->Get_Setting( 'max_execution_time' ) * 2 );

// Iterate through keys so we can grab references
		$keys = array_keys( $this->status['progress']['jobs'] );

		$ret = true;

		foreach ( $keys as $key ) {

			$job = & $this->status['progress']['jobs'][$key];

// Call the correct processor for this job
			switch ( $job['processor'] ) {

				case 'tables':
				case 'files':
				case 'transmission':
				case 'email':
					$processor = $this->Fetch_Processor( $job['processor'] );
					if ( ( $ret = $processor->Backup( $this, $this->stream, $this->status['progress'], $job ) ) !== true ) break 2;
					break;

				case 'reconstruct':
					if ( ( $ret = $this->Reconstruct( $job ) ) !== true ) break 2;
					break;

			}

// Job done - increase progress and drop the job
			$this->status['progress']['jobdone'] += $job['progresslen'];

			unset( $this->status['progress']['jobs'][$key] );

			$this->Tick();

		}

		return $ret;
	}

	/*private*/ function Reconstruct( & $job )
	{
// Flush all data
		if ( $job['progress'] == 0 ) {

			if ( ( $ret = $this->stream->Flush() ) !== true ) return $ret;

			$job['progress'] = 20;

			$this->Tick( false, true );

		}

// Close all files
		if ( $job['progress'] == 20 ) {

			if ( ( $ret = $this->stream->Close() ) !== true ) return $ret;

			$job['progress'] = 40;

			$this->Tick();

		}

// Prepare for reconstruction
		if ( $job['progress'] == 40 ) {

			if ( ( $ret = $this->stream->Start_Reconstruct() ) !== true ) return $ret;

			$job['progress'] = 60;

			$this->Tick();

		}

// Reconstruct any files that fragmented due to timeouts
		if ( $job['progress'] == 60 ) {

			while ( ( $ret = $this->stream->Do_Reconstruct() ) === true ) {

				$this->Tick();

			}

			if ( !is_array( $ret ) ) return $ret;

// Store the resulting file set
			$this->status['progress']['file_set'] = array_merge(
				$ret,
				array(
					'files'		=> $this->stream->Files(),
					'compressed'	=> $this->stream->Is_Compressed(),
					'encrypted'	=> $this->stream->Is_Encrypted(),
				)
			);

			$job['progress'] = 95;

			$this->Tick( false, true );

		}

// End reconstruction - remove any left temporary files etc
		if ( ( $ret = $this->stream->End_Reconstruct() ) !== true ) return $ret;

// All done, destroy the stream
		$this->stream = null;

		$job['progress'] = 100;

		return true;
	}

	/*public*/ function Process_Pull()
	{
// Check we have a backup running
		if ( $this->status['status'] != WPONLINEBACKUP_STATUS_RUNNING && $this->status['status'] != WPONLINEBACKUP_STATUS_TICKING ) return false;
		if ( $this->status['progress']['config']['target'] != 'online' ) return false;

// Check the nonce
		if ( $this->status['progress']['nonce'] == '' || $_GET['wponlinebackup_fetch'] != $this->status['progress']['nonce'] ) return false;

// Get parameters
		$which = array_key_exists( 'which', $_GET ) && strval( $_GET['which'] ) == 'data' ? 'data' : 'indx';
		$start = array_key_exists( 'start', $_GET ) ? intval( $_GET['start'] ) : 0;

		if ( $start > $this->status['progress']['file_set']['size'][$which] ) return false;

// Open the requested file
		if ( ( $f = @fopen( $this->status['progress']['file_set']['file'][$which], 'r' ) ) === false ) return false;

		if ( @fseek( $f, $this->status['progress']['file_set']['offset'][$which] + $start, SEEK_SET ) != 0 ) {
			@fclose( $f );
			return false;
		}

// Clear any data we have in any WordPress buffers - should not get much due to POST but just in case
		$cnt = ob_get_level();
		while ( $cnt-- > 0 ) ob_end_clean();

		header( 'Content-Length: ' . ( $this->status['progress']['file_set']['size'][$which] - $start + strlen( $this->status['progress']['nonce'] ) ) );

// Validation header
		echo 'OBFWRD' . $this->status['progress']['nonce'];

// Passthrough
		@fpassthru( $f );
		@fclose( $f );

// Capture any post-request junk - POST should have resolved most of this but double check
		ob_start( 'WPOnlineBackup_Capture_Junk' );

		return true;
	}
}

function WPOnlineBackup_Capture_Junk( $output )
{
	return '';
}

?>
