<?php

/*
WPOnlineBackup_Backup_Transmission - Handles all communication with the online vault
It will perform a synchronisation of the backup status at the start of the backup if we detect we are out of sync
After a backup, it will initiate and monitor transfer to the online vault.
Previous versions we ended backup and left it for server to fetch - now we wait for it to be fetched so we can update our internal status.
*/

define( 'WPONLINEBACKUP_SERVER', 'https://wordpress.backup-technology.com' );

define( 'API_ERROR_LOGIN_BASE',				0x00000100 );
define( 'API_ERROR_CUSTOM_BASE',			0x00001000 );

define( 'API_ERROR_LOGIN_FAILURE',			API_ERROR_LOGIN_BASE +		0x00001 );

define( 'API_ERROR_WPONLINEBACKUP_BASE',		API_ERROR_CUSTOM_BASE +		0x00000000 );

define( 'API_ERROR_WPONLINEBACKUP_KEY_MISMATCH',	API_ERROR_WPONLINEBACKUP_BASE +	0x00000003 );
define( 'API_ERROR_WPONLINEBACKUP_BLOG_NOT_FOUND',	API_ERROR_WPONLINEBACKUP_BASE +	0x00000004 );

class WPOnlineBackup_Backup_Transmission
{
	/*private*/ var $WPOnlineBackup;

	/*private*/ var $bootstrap;
	/*private*/ var $stream;
	/*private*/ var $progress;
	/*private*/ var $job;

	/*private*/ var $push_file;

	/*public*/ function WPOnlineBackup_Backup_Transmission( & $WPOnlineBackup )
	{
		require_once WPONLINEBACKUP_PATH . '/include/xml.php';

// Need formatting functions here
		require_once WPONLINEBACKUP_PATH . '/include/formatting.php';

		$this->WPOnlineBackup = & $WPOnlineBackup;
	}

	/*private*/ function Generate_Key_Hash()
	{
// Generate a hash of the key so we can validate it is still the same as the one used previously
		$type = $this->WPOnlineBackup->Get_Setting( 'encryption_type' );

		if ( $type == '' ) {
			$key = '';
		} else {
			$key = $this->WPOnlineBackup->Get_Setting( 'encryption_key' );
			$key = sha1( $type . str_pad( $key, 5120, $type . $key, STR_PAD_RIGHT ) );
			for ( $i = 0; $i <= 64; $i++ ) $key = sha1( $key );
		}

		return $key;
	}

	/*public*/ function Validate_Account()
	{
		// Validate_Account will return the following:
		//	true: if the account DOES have encryption configuration and the key matches
		//	0: if the account DOES have encryption configuration and the key DOES NOT match
		//	false: if the account has NO encryption configuration
		// 	or a generic transmission error

		// Generate query string with the key hash, but tell Status not to commit the key just yet, so we can still last minute change it
		// Normally we will only commit to it during Synchronise, which means a backup has been run, and it is from this point we don't allow key changing
		$q = array(
			'ka'		=> $this->Generate_Key_Hash(),
			'saveka'	=> 0,
		);

		if ( ( $ret = $this->Get( 'Status', $q, $xml, false ) ) !== true ) {

			if ( !is_numeric( $ret ) ) return $ret;

			if ( $xml[0] == API_ERROR_WPONLINEBACKUP_KEY_MISMATCH ) return 0;

			if ( $xml[0] == API_ERROR_LOGIN_FAILURE ) return sprintf( __( 'Failed to login to the online vault: %s' , 'wponlinebackup'), $xml[1] );

			return sprintf( __( 'An online request failed: The server responded with status code %d and error code %s: %s' , 'wponlinebackup'), $ret, $xml[0], $xml[1] );

		}

		if ( !isset( $xml->data->Status )
			|| !isset( $xml->data->Status[0]->KeySet[0]->_text )
			|| !isset( $xml->data->Status[0]->BSN[0]->_text )
			|| !isset( $xml->data->Status[0]->Items[0]->_text )
			|| !isset( $xml->data->Status[0]->Generations[0]->_text ) )
			return __( 'An online request failed: The server response was malformed. Please try again later.' , 'wponlinebackup');

		// Status will create the blog if it doesn't exist, so check if we have keys set
		if ( $xml->data->Status[0]->KeySet[0]->_text ) return true;

		return false;
	}

	/*public*/ function CleanUp( $ticking = false )
	{
	}

	/*public*/ function Backup( & $bootstrap, & $stream, & $progress, & $job )
	{
		$this->bootstrap = & $bootstrap;
		$this->stream = & $stream;
		$this->progress = & $progress;
		$this->job = & $job;

		switch ( $job['action'] ) {

			default:
			case 'synchronise':
				$ret = $this->Synchronise();
				break;

			case 'transmit':
				$ret = $this->Transmit();
				break;

		}

		return $ret;
	}

	/*private*/ function Synchronise()
	{
		global $wpdb;

		if ( $this->job['progress'] == 0 ) {

			$this->progress['message'] = __( 'Connecting to online backup vault...' , 'wponlinebackup');

// Generate query string
			$q = array(
				'ka'	=> $this->Generate_Key_Hash(),
			);

// Grab the backup serial number (BSN)
			if ( ( $ret = $this->Get( 'Status', $q, $xml ) ) !== true ) return $ret;

			if ( !isset( $xml->data->Status )
				|| !isset( $xml->data->Status[0]->BSN[0]->_text )
				|| !isset( $xml->data->Status[0]->Items[0]->_text )
				|| !isset( $xml->data->Status[0]->Generations[0]->_text ) )
				return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

// If it's the same as ours, we are fully in sync, set progress to 100 to move onto the backup stages
			if ( ( $this->progress['bsn'] = intval( $xml->data->Status[0]->BSN[0]->_text ) ) == get_option( 'wponlinebackup_bsn', '0' ) && get_option( 'wponlinebackup_in_sync', 0 ) ) {

				$this->job['progress'] = 100;

				return true;

			}

			$this->progress['message'] = __( 'Backup status is out of sync... Synchronising with server...' , 'wponlinebackup');

// We will start downloading the full file list from the server so we are in sync
			$this->job['total_items'] = intval( $xml->data->Status[0]->Items[0]->_text );
			$this->job['total_generations'] = intval( $xml->data->Status[0]->Generations[0]->_text );

			$this->job['progress'] = 1;

			$this->bootstrap->Tick();

		}

		if ( $this->job['progress'] == 1 ) {

			do {

				if ( ( $ret = $wpdb->query(
					'DELETE FROM `' . $wpdb->prefix . 'wponlinebackup_items` ' .
					'LIMIT 500'
				) ) === false ) return $this->bootstrap->DBError( __FILE__, __LINE__ );

				$this->bootstrap->Tick();

			} while ( $ret );

			$this->job['progress'] = 2;

		}

		if ( $this->job['progress'] == 2 ) {

			do {

				if ( ( $ret = $wpdb->query(
					'DELETE FROM `' . $wpdb->prefix . 'wponlinebackup_generations` ' .
					'LIMIT 500'
				) ) === false ) return $this->bootstrap->DBError( __FILE__, __LINE__ );

				$this->bootstrap->Tick();

			} while ( $ret );

			$this->job['progress'] = 5;

		}

		$segment_size = $this->WPOnlineBackup->Get_Setting( 'sync_segment_size' );

		while ( $this->job['progress'] < 53 ) {

// Download the file list for this serial number - we pass the serial number so we can detect a change in the middle of the synchronisation and abort
			$query = array(
				'bsn'		=> $this->progress['bsn'],
				'start'		=> $this->job['done_items'],
				'limit'		=> $segment_size,
			);

			if ( ( $ret = $this->Get( 'SynchroniseItems', $query, $xml ) ) !== true ) return $ret;

			if ( !isset( $xml->data->SynchroniseItems[0]->_attr->Final ) )
				return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

// If final attribute is 1, we are finished and can exit
			if ( $xml->data->SynchroniseItems[0]->_attr->Final == 1 ) {

				$this->job['progress'] = 53;

				$this->bootstrap->Tick();

				break;

			}

			if ( !isset( $xml->data->SynchroniseItems[0]->Item ) )
				return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

// Build the insert
			$inserts = array();

			foreach ( $xml->data->SynchroniseItems[0]->Item as $item ) {

				if ( !isset( $item->ID[0]->_text )
					|| !isset( $item->Bin[0]->_text )
					|| !isset( $item->ParentID[0]->_text )
					|| !isset( $item->Type[0]->_text )
					|| !isset( $item->Name[0]->_text )
					|| !isset( $item->Exists[0]->_text ) )
					return $this->bootstrap->MALError( __FILE__, __LINE__, $xml, 'Completed entries: ' . count( $inserts ) );

				$item->Bin[0]->_text = intval( $item->Bin[0]->_text );
				$item->ID[0]->_text = intval( $item->ID[0]->_text );
				$item->ParentID[0]->_text = intval( $item->ParentID[0]->_text );
				$item->Type[0]->_text = intval( $item->Type[0]->_text );
				$wpdb->escape_by_ref( $item->Name[0]->_text );
				$item->Exists[0]->_text = intval( $item->Exists[0]->_text );
				if ( isset( $item->FileSize[0]->_text ) ) $item->FileSize[0]->_text = intval( $item->FileSize[0]->_text );
				else $item->FileSize[0]->_text = 'NULL';
				if ( isset( $item->ModTime[0]->_text ) ) $item->ModTime[0]->_text = intval( $item->ModTime[0]->_text );
				else $item->ModTime[0]->_text = 'NULL';
				if ( isset( $item->Path[0]->_text ) ) $wpdb->escape_by_ref( $item->Path[0]->_text );
				else $item->Path[0]->_text = '';

				$inserts[] = '(' . $item->Bin[0]->_text . ', ' . $item->ID[0]->_text . ', ' . $item->ParentID[0]->_text . ', ' . $item->Type[0]->_text . ', ' .
						'\'' . $item->Name[0]->_text . '\', ' . $item->Exists[0]->_text . ', ' . $item->FileSize[0]->_text . ', ' .
						$item->ModTime[0]->_text . ', \'' . $item->Path[0]->_text . '\')';

			}

			$this->job['done_items'] += count( $inserts );

			$inserts = implode( ',', $inserts );

// Do the insert
			if ( $wpdb->query(
				'INSERT INTO `' . $wpdb->prefix . 'wponlinebackup_items` ' .
					'(bin, item_id, parent_id, type, name, `exists`, file_size, mod_time, path) ' .
				'VALUES ' . $inserts
			) === false )
				return $this->bootstrap->DBError( __FILE__, __LINE__ );

// Update the progress counter
			if ( $this->job['done_items'] >= $this->job['total_items'] ) $this->job['progress'] = 53;
			else {
				$this->job['progress'] = 5 + floor( ( $this->job['done_items'] * 48 ) / $this->job['total_items'] );
				if ( $this->job['progress'] >= 53 ) $this->job['progress'] = 53;
			}

			$this->bootstrap->Tick();

		}

		while ( $this->job['progress'] < 100 ) {

// Download the file list for this serial number - we pass the serial number so we can detect a change in the middle of the synchronisation and abort
			$query = array(
				'bsn'		=> $this->progress['bsn'],
				'start'		=> $this->job['done_generations'],
				'limit'		=> $segment_size,
			);

			if ( ( $ret = $this->Get( 'SynchroniseGenerations', $query, $xml ) ) !== true ) return $ret;

			if ( !isset( $xml->data->SynchroniseGenerations[0]->_attr->Final ) )
				return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

// If final attribute is 1, we are finished and can exit
			if ( $xml->data->SynchroniseGenerations[0]->_attr->Final == 1 ) {

				$this->job['progress'] = 100;

				$this->bootstrap->Tick();

				break;

			}

// Build the insert
			$inserts = array();

			if ( !isset( $xml->data->SynchroniseGenerations[0]->Generation ) )
				return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

			foreach ( $xml->data->SynchroniseGenerations[0]->Generation as $generation ) {

				if ( !isset( $generation->Bin[0]->_text )
					|| !isset( $generation->ID[0]->_text )
					|| !isset( $generation->BackupTime[0]->_text ) )
					return $this->bootstrap->MALError( __FILE__, __LINE__, $xml, 'Completed entries: ' . count( $inserts ) );

				$generation->Bin[0]->_text = intval( $generation->Bin[0]->_text );
				$generation->ID[0]->_text = intval( $generation->ID[0]->_text );
				$generation->BackupTime[0]->_text = intval( $generation->BackupTime[0]->_text );
				if ( isset( $generation->DeletedTime[0]->_text ) ) $generation->DeletedTime[0]->_text = intval( $generation->DeletedTime[0]->_text );
				else $generation->DeletedTime[0]->_text = 'NULL';
				if ( isset( $generation->FileSize[0]->_text ) ) $generation->FileSize[0]->_text = intval( $generation->FileSize[0]->_text );
				else $generation->FileSize[0]->_text = 'NULL';
				if ( isset( $generation->ModTime[0]->_text ) ) $generation->ModTime[0]->_text = intval( $generation->ModTime[0]->_text );
				else $generation->ModTime[0]->_text = 'NULL';

				$inserts[] = '(' . $generation->Bin[0]->_text . ', ' . $generation->ID[0]->_text . ', ' . $generation->BackupTime[0]->_text . ', ' .
					$generation->DeletedTime[0]->_text . ', ' . $generation->FileSize[0]->_text . ', ' . $generation->ModTime[0]->_text . ')';

			}

			$this->job['done_generations'] += count( $inserts );

			$inserts = implode( ',', $inserts );

// Do the insert
			if ( $wpdb->query(
				'INSERT INTO `' . $wpdb->prefix . 'wponlinebackup_generations` ' .
					'(bin, item_id, backup_time, deleted_time, file_size, mod_time) ' .
				'VALUES ' . $inserts
			) === false )
				return $this->bootstrap->DBError( __FILE__, __LINE__ );

// Update the progress counter
			if ( $this->job['done_generations'] >= $this->job['total_generations'] ) $this->job['progress'] = 100;
			else {
				$this->job['progress'] = 53 + floor( ( $this->job['done_generations'] * 47 ) / $this->job['total_generations'] );
				if ( $this->job['progress'] >= 100 ) $this->job['progress'] = 100;
			}

			$this->bootstrap->Tick();

		}

// Now in sync
		update_option( 'wponlinebackup_in_sync', '1' );
		update_option( 'wponlinebackup_bsn', $this->progress['bsn'] );

		return true;
	}

	/*private*/ function Transmit()
	{
		global $wpdb;

		if ( $this->job['progress'] == 0 ) {

// Generate random password for server to pull the backup with
			$this->progress['nonce'] = sha1( time() . serialize( $this->progress ) . 'online' );

// Grab filesizes
			$indx_size = $this->progress['file_set']['size']['indx'];
			$data_size = $this->progress['file_set']['size']['data'];
			$this->job['total'] = $indx_size + $data_size;

// Make the request for a pull
			$query = array(
				'bsn'		=> $this->progress['bsn'],
				'indx_size'	=> $indx_size,
				'data_size'	=> $data_size,
				'nonce'		=> $this->progress['nonce'],
				'start_time'	=> $this->progress['start_time'],
			);

			if ( ( $ret = $this->Get( 'Pull', $query, $xml ) ) !== true ) return $ret;

			if ( !isset( $xml->data->Pull )
				|| isset( $xml->data->Pull[0]->_text ) )
				return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

// Update progress
			$this->progress['message'] = sprintf( __( 'Transmitting the backup files to the server... (total size is %s)' , 'wponlinebackup'), WPOnlineBackup_Formatting::Fix_B( $this->job['total'], true ) );

			$this->job['progress'] = 5;

// Force an immedate tick - we'll check status on next schedule
			$this->bootstrap->Tick( true );

		}

		while ( $this->job['progress'] < 99 ) {

			$query = array(
				'bsn'		=> $this->progress['bsn'],
				'start'		=> $this->job['done_retention'],
			);

// Grab status from server
			if ( ( $ret = $this->Get( 'PullStatus', $query, $xml ) ) !== true ) return $ret;

			if ( !isset( $xml->data->PullStatus[0]->_attr->Complete ) )
				return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

			if ( $xml->data->PullStatus[0]->_attr->Complete == '1' ) {

				if ( isset( $xml->data->PullStatus[0]->_text ) )
					return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );


				if ( $this->job['progress'] < 80 ) {

// Processing - set to 80% and update message
					$this->job['progress'] = 80;

					$this->progress['message'] = __( 'Waiting while the server processes the backup data...' , 'wponlinebackup');

					$this->bootstrap->Log_Event(
						WPONLINEBACKUP_EVENT_INFORMATION,
						__( 'Transmission of the backup data to the online vault has completed.' , 'wponlinebackup')
					);

				}

// Force a tick that schedules the next run - no rush with server processing the backup
				$this->bootstrap->Tick( true );

			} else if ( $xml->data->PullStatus[0]->_attr->Complete == '2' ) {

				if ( $this->job['progress'] < 90 ) {

// Doing retention - set to 90% and update message
					$this->job['progress'] = 90;

					$this->progress['message'] = __( 'Performing retention...' , 'wponlinebackup');

					$this->bootstrap->Log_Event(
						WPONLINEBACKUP_EVENT_INFORMATION,
						__( 'The online vault has successfully processed the backup data.' , 'wponlinebackup')
					);

					$this->bootstrap->Tick();

				}

// Perform retention
				if ( !isset( $xml->data->PullStatus[0]->DeletedGeneration ) )
					return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

				$i = 0;

				foreach ( $xml->data->PullStatus[0]->DeletedGeneration as $gen ) {

					if ( !isset( $gen->_attr->Item )
						|| !isset( $gen->_attr->Size )
						|| !isset( $gen->_text ) )
						return $this->bootstrap->MALError( __FILE__, __LINE__, $xml, 'Iteration ' . $i );

// Do the delete
					if ( $wpdb->query(
						'DELETE FROM `' . $wpdb->prefix . 'wponlinebackup_generations` ' .
						'WHERE item_id = ' . intval( $gen->_attr->Item ) . ' AND backup_time = ' . intval( $gen->_text )
					) === false )
						return $this->bootstrap->DBError( __FILE__, __LINE__ );

					++$this->job['done_retention'];
					$this->job['retention_size'] += $gen->_attr->Size;

					$i++;

				}

// Tick
				$this->bootstrap->Tick();

			} else if ( $xml->data->PullStatus[0]->_attr->Complete == '3' ) {

// Grab new BSN
				if ( !isset( $xml->data->PullStatus[0]->BSN[0]->_text ) )
					return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

// Catch up with messages
				if ( $this->job['progress'] < 80 ) {

					$this->bootstrap->Log_Event(
						WPONLINEBACKUP_EVENT_INFORMATION,
						__( 'Transmission of the backup data to the online vault has completed.' , 'wponlinebackup')
					);

				}

				if ( $this->job['progress'] < 90 ) {

					$this->bootstrap->Log_Event(
						WPONLINEBACKUP_EVENT_INFORMATION,
						__( 'The online vault has successfully processed the backup data.' , 'wponlinebackup')
					);

				}

// Log the number of removed files and the total size
				if ( $this->job['done_retention'] )
					$this->bootstrap->Log_Event(
						WPONLINEBACKUP_EVENT_INFORMATION,
						sprintf( _n(
							'Retention completed; deleted %d file with a total size of %s, to reduce the online stored amount to below your maximum quota.',
							'Retention completed; deleted %d files with a total size of %s, to reduce the online stored amount to below your maximum quota.',
							$this->job['done_retention']
						, 'wponlinebackup'), $this->job['done_retention'], WPOnlineBackup_Formatting::Fix_B( $this->job['retention_size'], true ) )
					);
				else
					$this->bootstrap->Log_Event(
						WPONLINEBACKUP_EVENT_INFORMATION,
						__( 'Retention was not required, you are still within your maximum quota.' , 'wponlinebackup')
					);

				$this->job['new_bsn'] = $xml->data->PullStatus[0]->BSN[0]->_text;

				if ( $this->job['progress'] < 99 ) {

// Completed, set to 99% so we can run the PullComplete
					$this->job['progress'] = 99;

					$this->progress['message'] = __( 'Processing backup log...' , 'wponlinebackup');

				}

				$this->bootstrap->Tick();

			} else if ( $xml->data->PullStatus[0]->_attr->Complete == '4' ) {

// Error occurred on server
				if ( !isset( $xml->data->PullStatus[0]->Error )
					|| !isset( $xml->data->PullStatus[0]->Error[0]->_text ) )
					return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

				switch ( $xml->data->PullStatus[0]->Error[0]->_text ) {
					case 'ALREADY_IN_USE':
						return __( 'The server is already part way through receiving another backup. Please try again in a few moments.' , 'wponlinebackup');
					case 'CONNECT_FAILED':
						return __( 'The connection to your blog failed. This may be a temporary failure or your blog may not be accessible over the internet. Please try again later.' , 'wponlinebackup');
					case 'CONNECT_TIMEOUT':
						return __( 'The connection to your blog timed out part way through receiving the backup data. Please check your blog is not experiencing any network issues and try again later.' , 'wponlinebackup');
					case 'RETRIEVE_FAILED':
						return __( 'The server failed to retrieve data from your blog. Please check your blog is not experiencing any network issues and try again later.' , 'wponlinebackup');
					case 'EXCEEDS_QUOTA':
						return sprintf( __( 'The backup is larger than your complete quota on the online vault. It cannot be stored. Please reduce the backup size by excluding something - it is currently %s in size.' , 'wponlinebackup'), WPOnlineBackup_Formatting::Fix_B( array_sum( $this->progress['file_set']['size'] ), true ) );
					case 'JUNK':
						return __( 'The server attempted to retrieve the data, but received junk from your blog. This can happen if your blog is not accessible over the internet. Otherwise, you may have a third-party plugin installed that is changing the backup data as the server tries to receive it. Please contact support if this is the case so we may improve compatibility.' , 'wponlinebackup');
					case 'LOCKED':
						return __( 'The backup data on the server is currently locked. This is usually the case when a request has been made to delete the blog data and the server is still performing the deletion.' , 'wponlinebackup');
					case 'UNKNOWN_FAILURE':
				}

				$this->bootstrap->Log_Event(
					WPONLINEBACKUP_EVENT_ERROR,
					__( 'The server failed to retrieve the backup. The reason is unknown.' , 'wponlinebackup') . PHP_EOL .
						sprintf( __( 'The unknown error token the server returned was: %s' , 'wponlinebackup'), $xml->data->PullStatus[0]->Error[0]->_text )
				);

				return sprintf( __( 'The server failed to retrieve the backup. The reason is unknown. Please consult the activity log.' , 'wponlinebackup') );

			} else {

// Grab the status
				if ( !isset( $xml->data->PullStatus[0]->ReceivedIndx[0]->_text )
					|| !isset( $xml->data->PullStatus[0]->ReceivedData[0]->_text ) )
					return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

				$this->job['done'] = intval( $xml->data->PullStatus[0]->ReceivedIndx[0]->_text ) + intval( $xml->data->PullStatus[0]->ReceivedData[0]->_text );

// Update the progress
				$this->progress['message'] = sprintf( __( 'Transmitting the backup files to the server... (%s of %s transferred so far)' , 'wponlinebackup'), WPOnlineBackup_Formatting::Fix_B( $this->job['done'], true ), WPOnlineBackup_Formatting::Fix_B( $this->job['total'], true ) );

				if ( $this->job['done'] >= $this->job['total'] ) $this->job['progress'] = 79;
				else {
					$this->job['progress'] = 5 + floor( ( $this->job['done'] * 74 ) / $this->job['total'] );
					if ( $this->job['progress'] >= 100 ) $this->job['progress'] = 79;
				}

// Force a tick that schedules the next run - no rush with server getting the backup
				$this->bootstrap->Tick( true );

			}

		}

		if ( $this->job['progress'] == 99 ) {

			$query = array(
				'bsn'		=> $this->progress['bsn'],
			);

// Notify server we are done
			if ( ( $ret = $this->Get( 'PullComplete', $query, $xml ) ) !== true ) return $ret;

			if ( !isset( $xml->data->PullComplete ) )
				return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );

// Now cleanup deleted generations that have no entries anymore
			if ( $wpdb->query(
				'DELETE i FROM `' . $wpdb->prefix . 'wponlinebackup_items` i ' .
					'LEFT JOIN `' . $wpdb->prefix . 'wponlinebackup_generations` g ON (g.bin = i.bin AND g.item_id = i.item_id)' .
				'WHERE g.backup_time IS NULL'
			) === false )
				return $this->bootstrap->DBError( __FILE__, __LINE__ );

// Set status to 100%
			$this->job['status'] = 100;

			$this->progress['bsn'] = $this->job['new_bsn'];

		}

		return true;
	}

	/*private*/ function Get( $api, $data, & $xml, $in_backup = true )
	{
		return $this->Request( 'get', $api, $data, $xml, null, $in_backup );
	}

	/*private*/ function Request( $method, $api, $data, & $xml, $body, $in_backup )
	{
		$q = array();

		$api = urlencode( $api );

// Create the query string
		$data['blogurl']	= site_url( '/' );
		$data['username']	= $in_backup ? $this->progress['cache']['username'] : $this->WPOnlineBackup->Get_Setting( 'username' );
		$data['password']	= $in_backup ? $this->progress['cache']['password'] : $this->WPOnlineBackup->Get_Setting( 'password' );
		$data['version']	= WPONLINEBACKUP_VERSION;
		$data['lang']		= WPLANG;
		foreach ( $data as $key => $value )
			$q[] = urlencode( $key ) . '=' . urlencode( $value );
		$q = '?' . implode( '&', $q );

		if ( $method == 'get' ) $function = 'wp_remote_get';
		else $function = 'wp_remote_post';

// Make the request
		$response = $function(
			WPONLINEBACKUP_SERVER . '/API/' . $api . $q,
			array(
				'timeout'	=> 30,
				'sslverify'	=> !$this->WPOnlineBackup->Get_Setting( 'ignore_ssl_cert' ),
				'body'		=> $body,
			)
		);

		if ( is_wp_error( $response ) ) {

			if ( $in_backup )
				return $this->bootstrap->COMError(
					__FILE__, __LINE__,
					sprintf( __( 'A request failed for API %s.' . PHP_EOL . 'Please ensure your blog is able to perform HTTPS requests to %s. You may need to contact your web host regarding this.' , 'wponlinebackup'), $api, WPONLINEBACKUP_SERVER ) . PHP_EOL .
						OBFW_Exception_WP( $response ),
					__( 'Communication with the online backup vault failed; more information can be found in the Activity Log.' , 'wponlinebackup')
				);
			else
				return sprintf( __( 'Communication with the online backup vault failed. Please ensure your blog is able to perform HTTPS requests to %s. You may need to contact your web host regarding this. The error was: %s.' , 'wponlinebackup'), WPONLINEBACKUP_SERVER, OBFW_Exception_WP( $response ) );

		}

// Parse the response
		$xml = new WPOnlineBackup_XML();

		if ( ( $ret = $xml->fetch( $response['body'] ) ) !== true ) {

			if ( $in_backup )
				return $this->bootstrap->MALError( __FILE__, __LINE__, $xml, $ret );
			else
				return $ret;

		}

		if ( $response['response']['code'] == 200 ) return true;

// Parse the error response
		if ( !isset( $xml->data->Error )
			|| !isset( $xml->data->Error[0]->Number )
			|| !isset( $xml->data->Error[0]->Message ) ) {

			if ( $in_backup )
				return $this->bootstrap->MALError( __FILE__, __LINE__, $xml );
			else
				return __( 'An online request failed: The server response was malformed. Please try again later.' , 'wponlinebackup');

		}

		$number = isset( $xml->data->Error[0]->Number[0]->_text ) ? $xml->data->Error[0]->Number[0]->_text : 'Unknown';
		$message = isset( $xml->data->Error[0]->Message[0]->_text ) ? $xml->data->Error[0]->Message[0]->_text : 'Unknown';

		if ( $in_backup )
			return $this->bootstrap->COMError(
				__FILE__, __LINE__,
				sprintf( __( 'An online request failed: The server responded with status code %d and error code %s: %s' , 'wponlinebackup'), $response['response']['code'], $number, $message ),
				sprintf( __( 'The server returned the following error message: %s' , 'wponlinebackup'), $message )
			);
		else {
			$xml = array( $number, $message );

			return $response['response']['code'];
		}
	}
}

?>
