<?php
/**
 * WooCommerce Customer/Order CSV Export
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Customer/Order CSV Export to newer
 * versions in the future. If you wish to customize WooCommerce Customer/Order CSV Export for your
 * needs please refer to http://docs.woocommerce.com/document/ordercustomer-csv-exporter/
 *
 * @package     WC-Customer-Order-CSV-Export/Background-Export
 * @author      SkyVerge
 * @copyright   Copyright (c) 2012-2017, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

defined( 'ABSPATH' ) or exit;

/**
 * Customer/Order CSV Export Background Export
 *
 * Class that handles performing exports in the background
 *
 * @since 4.0.0
 */
class WC_Customer_Order_CSV_Export_Background_Export extends SV_WP_Background_Job_Handler {


	/**
	 * Initialize background export handler
	 *
	 * @since 4.0.0
	 */
	public function __construct() {

		$this->prefix   = 'wc_customer_order_csv_export';
		$this->action   = 'background_export';
		$this->data_key = 'object_ids';

		parent::__construct();

		add_action( "{$this->identifier}_job_complete", array( $this, 'finish_export' ) );
		add_action( "{$this->identifier}_job_failed",   array( $this, 'failed_export' ) );
		add_action( "{$this->identifier}_job_deleted",  array( $this, 'delete_export_data' ) );
	}


	/**
	 * Finish the export job
	 *
	 * @since 4.0.0
	 * @param object $job Job instance
	 */
	public function finish_export( $job ) {

		wc_customer_order_csv_export()->get_export_handler_instance()->finish_export( $job );
	}


	/**
	 * Process a single item from the job
	 *
	 * @since 4.0.0
	 * @param mixed $item
	 * @param object $job related job instance
	 */
	public function process_item( $item, $job ) {

		wc_customer_order_csv_export()->get_export_handler_instance()->export_item( $item, $job );
	}


	/**
	 * Handle failed  export jobs
	 *
	 * @since 4.0.0
	 * @param object $job Job instance
	 */
	public function failed_export( $job ) {

		wc_customer_order_csv_export()->get_export_handler_instance()->failed_export( $job );
	}


	/**
	 * Delete data & files related to the export
	 *
	 * @since 4.0.0
	 * @param stdClass $job
	 */
	public function delete_export_data( $job ) {

		// Delete exported file generated by the job that was deleted
		// suppress errors as they're not really helpful here
		@unlink( $job->file_path );

		// delete user notices related to the export
		if ( $job->created_by ) {
			wc_customer_order_csv_export()->get_export_handler_instance()->remove_export_finished_notice( $job->id, $job->created_by );
		}
	}

}
