<?php

/**
 * WC_Gateway_Stripe_Connect class.
 *
 * @extends WC_Payment_Gateway
 */
class WC_Gateway_Stripe_Connect extends WC_Payment_Gateway {


	/** @var bool Whether or not logging is enabled */
	public static $log_enabled = false;

	/** @var WC_Logger Logger instance */
	public static $log = false;

	/**
	 * This is a list of non decimal currencies that supported by stripe
	 * find the full list here : https://support.stripe.com/questions/which-zero-decimal-currencies-does-stripe-support
	 */
	public $non_decimal_currencies = array( 'BIF','CLP' ,'DJF' ,'GNF' ,'JPY' ,'KMF' ,'KRW' ,'MGA' ,'PYG' ,'RWF' ,'VND' ,'VUV' ,'XAF' ,'XOF' ,'XPF' ); 

	function __construct() {

		$this->id           = 'stripe-connect';
		$this->method_title = __( 'Stripe Connect', 'wcv_stripe_connect' );
		// 1.0.2 Stripe supports V/M/A for everyone, but D/J for USA, so show a different icon for USD currency than all others
		$this->icon         = plugins_url( '/assets/images/cards-outside-usa.png', dirname( __FILE__ ) );
		if ( in_array( get_option( 'woocommerce_currency' ), array( 'USD' ) ) )
            $this->icon         = plugins_url( '/assets/images/cards.png', dirname( __FILE__ ) );
		$this->has_fields   = true;
		$this->api_endpoint = 'https://api.stripe.com/';
		$this->supports     = array(
			'subscriptions',
			'products',
			'subscription_cancellation',
			'subscription_reactivation',
			'subscription_suspension',
			'subscription_amount_changes',
			'subscription_date_changes'
		);

		// Load the form fields
		$this->init_form_fields();

		// Load the settings.
		$this->init_settings();

		// Get setting values
		$this->title     	  		= $this->settings['title'];
		$this->description    		= $this->settings['description'];
		$this->enabled    	  		= $this->settings['enabled'];
		$this->testmode       		= $this->settings['testmode'];
		$this->stripe_checkout   	= isset( $this->settings['stripe_checkout'] ) && $this->settings['stripe_checkout'] == 'yes' ? true : false;

		$this->secret_key    		= $this->testmode == 'no' ? $this->settings['secret_key'] : $this->settings['test_secret_key'];
		$this->publishable_key  	= $this->testmode == 'no' ? $this->settings['publishable_key'] : $this->settings['test_publishable_key'];
		$this->enable_logging 		= ( isset( $this->settings['enable_logging'] ) ) ? $this->settings['enable_logging'] : false; 

		self::$log_enabled    		= $this->enable_logging; 

		// This is used to calculate decimal or non decimal multipliers 
		$this->multiplier 			= ( in_array( get_woocommerce_currency() , $this->non_decimal_currencies ) ) ? 1 : 100; 
		$this->minimum_order_total  = ( $this->multiplier == 1 ) ? 0.50 : 50.00; 

		// Hooks
		add_action( 'wp_enqueue_scripts', array( &$this, 'payment_scripts' ) );
		add_action( 'admin_notices', array( &$this, 'checks' ) );
		add_action( 'woocommerce_update_options_payment_gateways', array( &$this, 'process_admin_options' ) );
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
	}

	/**
	 * Logging method.
	 * @param string $message
	*/
	public static function log( $message ) {
		
		if ( self::$log_enabled ) {
			if ( empty( self::$log ) ) {
				self::$log = new WC_Logger();
			}
			self::$log->add( 'wc-vendors-gateway-stripe-connect', $message );
		}

	} // log() 


	/**
	 * Check if SSL is enabled and notify the user
	 * */
	function checks() {
		global $woocommerce;

		if ( $this->enabled == 'no' )
			return;

		// Version check
		if ( $woocommerce->version < '1.5.8' ) {

			echo '<div class="error"><p>' . __( 'Stripe now uses stripe.js for security and requires WooCommerce 1.5.8. Please update WooCommerce to continue using Stripe.', 'wcv_stripe_connect' ) . '</p></div>';

			return;
		}

		// Check required fields
		if ( ! $this->secret_key ) {

			echo '<div class="error"><p>' . sprintf( __( 'Stripe error: Please enter your secret key <a href="%s">here</a>', 'wcv_stripe_connect' ), admin_url( 'admin.php?page=wc-settings&tab=checkout&section=wc_gateway_stripe_connect' ) ) . '</p></div>';

			return;

		} elseif ( ! $this->publishable_key ) {

			echo '<div class="error"><p>' . sprintf( __( 'Stripe error: Please enter your publishable key <a href="%s">here</a>', 'wcv_stripe_connect' ), admin_url( 'admin.php?page=wc-settings&tab=checkout&section=wc_gateway_stripe_connect' ) ) . '</p></div>';

			return;
		}

		// Simple check for duplicate keys
		if ( $this->secret_key == $this->publishable_key ) {

			echo '<div class="error"><p>' . sprintf( __( 'Stripe error: Your secret and publishable keys match. Please check and re-enter.', 'wcv_stripe_connect' ), admin_url( 'admin.php?page=wc-settings&tab=checkout&section=wc_gateway_stripe_connect' ) ) . '</p></div>';

			return;
		}

		// Show message if enabled and FORCE SSL is disabled and WordpressHTTPS plugin is not detected
		if ( get_option( 'woocommerce_force_ssl_checkout' ) == 'no' && ! class_exists( 'WordPressHTTPS' ) ) {

			echo '<div class="error"><p>' . sprintf( __( 'Stripe is enabled, but the <a href="%s">force SSL option</a> is disabled; your checkout may not be secure! Please enable SSL and ensure your server has a valid SSL certificate - Stripe will only work in test mode.', 'wcv_stripe_connect' ), admin_url( 'admin.php?page=wc-settings&tab=checkout' ) ) . '</p></div>';

		}
	}

	/**
	 * Check if this gateway is enabled and available in the user's country
	 */
	function is_available() {
		global $woocommerce;

		if ( $this->enabled == "yes" ) {

			if ( $woocommerce->version < '1.5.8' )
				return false;

			if ( ! is_ssl() && $this->testmode != 'yes' )
				return false;

			// Currency Check disabled in v1.0.1 as Stripe now has 135 currency symbols as supported, no longer limited to USD/CAD/GBP -Ben
            //if ( ! in_array( get_option( 'woocommerce_currency' ), array( 'USD', 'CAD', 'GBP' ) ) )
                //return false;

			// Required fields check
			if ( ! $this->secret_key ) return false;
			if ( ! $this->publishable_key ) return false;

			return true;
		}

		return false;
	}

	/**
	 * Initialise Gateway Settings Form Fields
	 */
	function init_form_fields() {

		$this->form_fields = array(
			'enabled' => array(
				'title' => __( 'Enable/Disable', 'wcv_stripe_connect' ),
				'label' => __( 'Enable Stripe', 'wcv_stripe_connect' ),
				'type' => 'checkbox',
				'description' => '',
				'default' => 'no'
			),
			'title' => array(
				'title' => __( 'Title', 'wcv_stripe_connect' ),
				'type' => 'text',
				'description' => __( 'This controls the title which the user sees during checkout.', 'wcv_stripe_connect' ),
				'default' => __( 'Credit Card (Stripe)', 'wcv_stripe_connect' )
			),
			'description' => array(
				'title' => __( 'Description', 'wcv_stripe_connect' ),
				'type' => 'textarea',
				'description' => __( 'This controls the description which the user sees during checkout.', 'wcv_stripe_connect' ),
				'default' => 'Pay with your credit card via Stripe.'
			),
			'testmode' => array(
				'title' => __( 'Test mode', 'wcv_stripe_connect' ),
				'label' => __( 'Enable Test Mode', 'wcv_stripe_connect' ),
				'type' => 'checkbox',
				'description' => __( 'Place the payment gateway in test mode using test API keys.', 'wcv_stripe_connect' ),
				'default' => 'yes'
			),
			'stripe_checkout' => array(
				'title' => __( 'Stripe Checkout', 'wcv_stripe_connect' ),
				'label' => __( 'Enable Stripe Checkout', 'wcv_stripe_connect' ),
				'type' => 'checkbox',
				'description' => __( 'If enabled, this option shows a "pay" button and modal credit card form on the checkout, instead of credit card fields directly on the page.', 'wcv_stripe_connect' ),
				'default' => 'no'
			),
			'live-credentials-title' => array(
				'title' => __( 'Live credentials', 'wcv_stripe_connect' ),
				'type' => 'title',
			),
			'secret_key' => array(
				'title' => __( 'Secret Key', 'wcv_stripe_connect' ),
				'type' => 'text',
				'description' => __( 'Get your API keys from your stripe account.', 'wcv_stripe_connect' ),
				'default' => ''
			),
			'publishable_key' => array(
				'title' => __( 'Publishable Key', 'wcv_stripe_connect' ),
				'type' => 'text',
				'description' => __( 'Get your API keys from your stripe account.', 'wcv_stripe_connect' ),
				'default' => ''
			),
			'client_id' => array(
				'title' => __( 'Client ID', 'wcv_stripe_connect' ),
				'type' => 'text',
				'description' => __( 'Get your client ID from your stripe account, the Apps menu.', 'wcv_stripe_connect' ),
				'default' => ''
			),
			'test-credentials-title' => array(
				'title' => __( 'Test credentials', 'wcv_stripe_connect' ),
				'type' => 'title',
			),
			'test_secret_key' => array(
				'title' => __( 'Test Secret Key', 'wcv_stripe_connect' ),
				'type' => 'text',
				'description' => __( 'Get your API keys from your stripe account.', 'wcv_stripe_connect' ),
				'default' => ''
			),
			'test_publishable_key' => array(
				'title' => __( 'Test Publishable Key', 'wcv_stripe_connect' ),
				'type' => 'text',
				'description' => __( 'Get your API keys from your stripe account.', 'wcv_stripe_connect' ),
				'default' => ''
			),
			'test_client_id' => array(
				'title' => __( 'Test Client ID', 'wcv_stripe_connect' ),
				'type' => 'text',
				'description' => __( 'Get your client ID from your stripe account, the Apps menu.', 'wcv_stripe_connect' ),
				'default' => ''
			),
			'enable_logging' => array(
				'title' => __( 'Enable Logging', 'wcv_stripe_connect' ),
				'label' => __( 'Enable Logging', 'wcv_stripe_connect' ),
				'type' => 'checkbox',
				'description' => __( 'Enable writing to a log for for debugging. This is found in WooCommerce > System Status > Logs', 'wcv_stripe_connect' ),
				'default' => 'no'
			),
		);
	}

	/**
	 * Admin Panel Options
	 * - Options for bits like 'title' and availability on a country-by-country basis
	 */
function admin_options() {
        ?>

        <h3><?php _e( 'Stripe Connect', 'wcv_stripe_connect' ); ?></h3>
        <p><?php _e( 'Stripe works by adding credit card fields on the checkout and then sending the details to Stripe for verification.', 'wcv_stripe_connect' ); ?></p>
                <table class="form-table">
                        <?php $this->generate_settings_html(); ?>
                        </table><!--/.form-table-->
        <?php
        }

	/**
	 * Payment form on checkout page
	 */
	function payment_fields() {
		global $woocommerce;
?>
		<fieldset>

			<?php if ( $this->description ) : ?>
				<p><?php echo $this->description; ?>
					<?php if ( $this->testmode == 'yes' ) : ?>
						<?php _e( 'TEST MODE ENABLED. In test mode, you can use the card number 4242424242424242 with any CVC and a valid expiration date.', 'wcv_stripe_connect' ); ?>
					<?php endif; ?></p>
			<?php endif; ?>

			<?php if ( is_user_logged_in() && ( $credit_cards = get_user_meta( get_current_user_id(), '_stripe_customer_id', false ) ) ) : ?>
				<p class="form-row form-row-wide">

					<a class="button" style="float:right;" href="<?php echo get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ); ?>#saved-cards"><?php _e( 'Manage cards', 'wcv_stripe_connect' ); ?></a>

					<?php foreach ( $credit_cards as $i => $credit_card ) : ?>
						<input type="radio" id="stripe_card_<?php echo $i; ?>" name="stripe_customer_id" style="width:auto;" value="<?php echo $i; ?>" />
						<label style="display:inline;" for="stripe_card_<?php echo $i; ?>"><?php _e( 'Card ending with', 'wcv_stripe_connect' ); ?> <?php echo $credit_card['active_card']; ?> (<?php echo $credit_card['exp_month'] . '/' . $credit_card['exp_year'] ?>)</label><br />
					<?php endforeach; ?>

					<input type="radio" id="new" name="stripe_customer_id" style="width:auto;" <?php checked( 1, 1 ) ?> value="new" /> <label style="display:inline;" for="new"><?php _e( 'Use a new credit card', 'wcv_stripe_connect' ); ?></label>

				</p>
				<div class="clear"></div>
			<?php endif; ?>

			<div class="stripe_new_card">

				<?php if ( $this->stripe_checkout ) : ?>

					<a id="stripe_payment_button" class="button" href="#"
						data-description=""
						data-amount="<?php echo $woocommerce->cart->total * $this->multiplier; ?>"
						data-name="<?php echo sprintf( __( '%s', 'woocommerce' ), get_bloginfo( 'name' ) ); ?>"
						data-label="<?php _e( 'Confirm and Pay', 'woocommerce' ); ?>"
						data-currency="<?php echo strtolower( get_woocommerce_currency() ); ?>"
						><?php _e( 'Enter payment details', 'woocommerce' ); ?></a>

				<?php else : ?>

					<p class="form-row form-row-wide">
						<label for="stripe_cart_number"><?php _e( "Credit Card number", 'wcv_stripe_connect' ) ?> <span class="required">*</span></label>
						<input type="text" autocomplete="off" class="input-text card-number" />
					</p>
					<div class="clear"></div>
					<p class="form-row form-row-first">
						<label for="cc-expire-month"><?php _e( "Expiration date", 'wcv_stripe_connect' ) ?> <span class="required">*</span></label>
						<select id="cc-expire-month" class="woocommerce-select woocommerce-cc-month card-expiry-month">
							<option value=""><?php _e( 'Month', 'wcv_stripe_connect' ) ?></option>
							<?php
			$months = array();
		for ( $i = 1; $i <= 12; $i++ ) :
			$timestamp = mktime( 0, 0, 0, $i, 1 );
		$months[date( 'n', $timestamp )] = date( 'F', $timestamp );
		endfor;
		foreach ( $months as $num => $name ) printf( '<option value="%u">%s</option>', $num, $name );
?>
						</select>
						<select id="cc-expire-year" class="woocommerce-select woocommerce-cc-year card-expiry-year">
							<option value=""><?php _e( 'Year', 'wcv_stripe_connect' ) ?></option>
							<?php
		for ( $i = date( 'y' ); $i <= date( 'y' ) + 15; $i++ ) printf( '<option value="20%u">20%u</option>', $i, $i );
?>
						</select>
					</p>
					<p class="form-row form-row-last">
						<label for="stripe_card_csc"><?php _e( "Card security code", 'wcv_stripe_connect' ) ?> <span class="required">*</span></label>
						<input type="text" id="stripe_card_csc" maxlength="4" style="width:4em;" autocomplete="off" class="input-text card-cvc" />
						<span class="help stripe_card_csc_description"></span>
					</p>
					<div class="clear"></div>

				<?php endif; ?>

			</div>

		</fieldset>
		<?php
	}

	/**
	 * payment_scripts function.
	 *
	 * Outputs scripts used for stripe payment
	 *
	 * @access public
	 */
	function payment_scripts() {

		if ( ! is_checkout() )
			return;

		if ( $this->stripe_checkout ) {

			wp_enqueue_script( 'stripe', 'https://checkout.stripe.com/v2/checkout.js', '', '2.0', true );
			wp_enqueue_script( 'woocommerce_stripe', plugins_url( 'assets/js/stripe_checkout.js', dirname( __FILE__ ) ), array( 'stripe' ), '2.0', true );

		} else {

			wp_enqueue_script( 'stripe', 'https://js.stripe.com/v1/', '', '1.0', true );
			wp_enqueue_script( 'woocommerce_stripe', plugins_url( 'assets/js/stripe.js', dirname( __FILE__ ) ), array( 'stripe' ), '1.0', true );

		}

		$stripe_params = array(
			'key' => $this->publishable_key
		);

		// If we're on the pay page we need to pass stripe.js the address of the order.
		if ( !empty( $_GET['order_id'] ) && is_page( woocommerce_get_page_id( 'pay' ) ) ) {
			$order_key = urldecode( $_GET['order'] );
			$order_id = (int) $_GET['order_id'];
			$order = new WC_Order( $order_id );
			$_order_id 	= ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->id : $order->get_id();  	

			if ( $_order_id == $order_id && $order->get_order_key() == $order_key ) {

				$billing_first_name = ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_first_name : $order->get_billing_first_name(); 
				$billing_last_name 	= ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_last_name : $order->get_billing_last_name(); 
				$billing_address_1  = ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_address_1 : $order->get_billing_address_1(); 
				$billing_address_2  = ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_address_2 : $order->get_billing_address_2(); 
				$billing_state		= ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_state : $order->get_billing_state() ; 
				$billing_city		= ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_city : $order->get_billing_city();
				$billing_postcode	= ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_postcode : $order->get_billing_postcode();	
				$billing_country	= ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_country : $order->get_billing_country();	

				$stripe_params['billing_first_name'] = $billing_first_name;
				$stripe_params['billing_last_name']  = $billing_last_name;
				$stripe_params['billing_address_1']  = $billing_address_1;
				$stripe_params['billing_address_2']  = $billing_address_2;
				$stripe_params['billing_state']      = $billing_state;
				$stripe_params['billing_city']       = $billing_city;
				$stripe_params['billing_postcode']   = $billing_postcode;
				$stripe_params['billing_country']    = $billing_country;
			}
		}

		wp_localize_script( 'woocommerce_stripe', 'wcv_stripe_connect_params', $stripe_params );
	}

	/**
	 * Process the payment
	 */
	function process_payment( $order_id ) {
		global $woocommerce;

		$customer_id  = 0;
		$vendors_paid = array(); 
		$order        = new WC_Order( $order_id );
		$stripe_token = isset( $_POST['stripe_token'] ) ? woocommerce_clean( $_POST['stripe_token'] ) : '';

		WC_Gateway_Stripe_Connect::log( '===========================================================================================' . $order_id ); 
		WC_Gateway_Stripe_Connect::log( 'Processing Order: ' . $order_id ); 

		try {
			require_once 'lib/Stripe.php';
			Stripe::setApiKey( $this->secret_key );

			// Check if paying via customer ID
			if ( isset( $_POST['stripe_customer_id'] ) && $_POST['stripe_customer_id'] !== 'new' && is_user_logged_in() ) {
				$customer_ids = get_user_meta( get_current_user_id(), '_stripe_customer_id', false );

				if ( isset( $customer_ids[ $_POST['stripe_customer_id'] ]['customer_id'] ) ){ 
					$customer_id = $customer_ids[ $_POST['stripe_customer_id'] ]['customer_id'];
					WC_Gateway_Stripe_Connect::log( sprintf( __( 'Customer ID: %s', 'wcv_stripe_connect' ), $customer_id ) ); 
				}
				else {
					WC_Gateway_Stripe_Connect::log( __( 'Invalid card.', 'wcv_stripe_connect' ) ); 
					throw new Exception( __( 'Invalid card.', 'wcv_stripe_connect' ) );
				}
			}

			// Else, Check token
			else if ( empty( $stripe_token ) ) {
					WC_Gateway_Stripe_Connect::log( __( 'Please make sure your card details have been entered correctly and that your browser supports JavaScript.', 'wcv_stripe_connect' ) ); 
					throw new Exception( __( 'Please make sure your card details have been entered correctly and that your browser supports JavaScript.', 'wcv_stripe_connect' ) );
				}

			// Check amount
			$orer_total 		= ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->get_order_total() : $order->get_total(); 	

			if ( $orer_total  * $this->multiplier < $this->minimum_order_total ) {
				WC_Gateway_Stripe_Connect::log( __( 'Minimum order total is 0.50', 'wcv_stripe_connect' ) );
				throw new Exception( __( 'Minimum order total is 0.50', 'wcv_stripe_connect' ) );
			}

			// Save token if logged in
			if ( ! $customer_id && $stripe_token ) {
				$customer_id = $this->add_customer( $order, $stripe_token );
			}

			// Static data
			$order_data = array(
				"currency"    => strtolower( get_woocommerce_currency() ),
				"description" => sprintf( __( '%s - Order %s', 'wp_stripe' ), esc_html( get_bloginfo( 'name' ) ), $order->get_order_number() ),
			);

			$charge_ids = array();

			// Process the vendor charges first
			$receivers 		= WCV_Vendors::get_vendor_dues_from_order( $order, false );
			WC_Gateway_Stripe_Connect::log( sprintf( __( 'Order Dues %s', 'wcv_stripe_connect' ), print_r( $receivers, true ) ) ); 

			foreach ( $receivers as $vendor_id => $products ) {

				if ( $vendor_id == 1 ) continue;

				$access_token = get_user_meta( $vendor_id, '_stripe_connect_access_key', true );

				// If they haven't connected to stripe do nothing 
				if ( empty( $access_token ) ) continue;

				$product_admin_fee = 0; 
				$total_admin_fee = 0;
				$total = 0;

				foreach ( $products as $key => $product ) {

					$product_admin_fee 	= $receivers[1][ $key ][ 'total' ]; 
					$total_admin_fee 	+= $receivers[1][ $key ][ 'total' ];
					$total 				+= ( $product[ 'total' ] + $product_admin_fee );
					unset( $receivers[ 1 ][ $key ], $receivers[ $vendor_id ][ $key ] );
					$vendors_paid[ $vendor_id ][] = array( 'product_id' => $product[ 'product_id' ], 'order_id' => $order_id ); 

				}	

				$token 								= Stripe_Token::create( array( 'customer' => $customer_id ), $access_token );
				$order_data[ 'amount' ] 			= round( $total, 2 ) * $this->multiplier;
				$order_data[ 'application_fee' ] 	= round( $total_admin_fee, 2 ) * $this->multiplier;
				$order_data[ 'card' ] 				= !empty( $token->id ) ? $token->id : $stripe_token;

				$order_data = apply_filters( 'wcv_stripe_order_data', $order_data ); 

				WC_Gateway_Stripe_Connect::log( sprintf( __( 'Vendor order data %s ', 'wcv_stripe_connect' ), print_r( $order_data, true ) ) ); 

				$charge 		= Stripe_Charge::create( $order_data, $access_token );
				$charge_ids[] 	= $charge->id;

				do_action( 'wcv_stripe_change_id_generated', $order_id, $charge->id );

			}

			// Now process any remaining admin charges for when the vendor hasn't connected stripe or have stripe. 
			$total = 0; 
			foreach ( $receivers as $vendor_id => $products ) {

				foreach ( $products as $product_id => $product ) {
					$total += $product['total'];
					unset( $receivers[$vendor_id][$product_id] );
				}
			}

			unset( $order_data['card'], $order_data['customer'], $order_data['application_fee'] );

			$order_data['amount'] = round( $total, 2 ) * $this->multiplier; 

			// Only output the admin dues if there are any 
			if ( isset( $receivers ) ) WC_Gateway_Stripe_Connect::log( sprintf( __( 'Admin dues order data %s ', 'wcv_stripe_connect' ), print_r( $order_data, true ) ) ); 

			if ( ! empty( $order_data['amount'] ) ) {

				if ( $customer_id ) {
					$order_data['customer'] = $customer_id;
				} else {
					$order_data['card'] = $stripe_token;
				}

				$charge = Stripe_Charge::create( $order_data ); 
				$charge_ids[] = $charge->id;
			}

			$receivers = array_filter( $receivers );

		} catch( Exception $e ) {

			WC_Gateway_Stripe_Connect::log( sprintf( __( 'Error: %s ', 'wcv_stripe_connect' ), $e->getMessage() ) );
			wc_add_notice( __( 'Error: ', 'wcv_stripe_connect' ) . $e->getMessage(), 'error' );
			return;
		}

		// Add order note
		$order->add_order_note( sprintf( __( 'Stripe payment completed (Charge ID: %s)', 'wcv_stripe_connect' ), implode( ', ', $charge_ids ) ) );

		// Payment complete
		$order->payment_complete();

		if ( class_exists( 'WCV_Commission' ) ) {

			foreach ( $vendors_paid as $vendor_id => $products ) {

				foreach ( $products as $product ) {
					WCV_Commission::set_vendor_product_commission_paid( $vendor_id, $product['product_id'], $product['order_id'] ); 
				}

			}

		}

		// Remove cart
		$woocommerce->cart->empty_cart();

		WC_Gateway_Stripe_Connect::log( '===========================================================================================' . $order_id ); 

		// Return thank you page redirect
		return array(
			'result'  => 'success',
			'redirect' => $this->get_return_url( $order )
		);

	}


	/**
	 * add_customer function.
	 *
	 * @access public
	 * @param mixed   $stripe_token
	 * @return void
	 */
	function add_customer( $order, $stripe_token ) {

		if ( ! $stripe_token ) {
			return;
		}

		require_once 'lib/Stripe.php';
		Stripe::setApiKey( $this->secret_key );

		$billing_email 		= ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_email : $order->get_billing_email(); 
		$shipping_first_name = ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_first_name 	: $order->get_shipping_first_name(); 
		$shipping_last_name  = ( version_compare( WC_VERSION, '2.7', '<' ) ) ? $order->billing_last_name 	: $order->get_shipping_last_name(); 

		$customer = Stripe_Customer::create( array(
				'email'       => $billing_email,
				'description' => 'Customer: ' . $shipping_first_name . ' ' . $shipping_last_name,
				"card"        => $stripe_token
			)
		);

		add_user_meta( get_current_user_id(), '_stripe_customer_id', array(
				'customer_id' => $customer->id,
				'active_card' => !empty( $customer->sources->data[0]->last4 ) ? 	$customer->sources->data[0]->last4 : '',
				'exp_year'    => !empty( $customer->sources->data[0]->exp_year ) ?	$customer->sources->data[0]->exp_year : '',
				'exp_month'   => !empty( $customer->sources->data[0]->exp_month ) ? $customer->sources->data[0]->exp_month : '',
			) );

		return $customer->id;
	}

}