/**
 * Tiered Commission
 * https://www.wcvendors.com/
 */

(function ($) {
	$.fn.serializeObject = function () {

		var self = this,
			json = {},
			push_counters = {},
			patterns = {
				"validate": /^[a-zA-Z][a-zA-Z0-9_]*(?:\[(?:\d*|[a-zA-Z0-9_]+)\])*$/,
				"key": /[a-zA-Z0-9_]+|(?=\[\])/g,
				"push": /^$/,
				"fixed": /^\d+$/,
				"named": /^[a-zA-Z0-9_]+$/
			};


		this.build = function (base, key, value) {
			base[key] = value;
			return base;
		};

		this.push_counter = function (key) {
			if ( push_counters[key] === undefined ) {
				push_counters[key] = 0;
			}
			return push_counters[key] ++;
		};

		$.each($(this).serializeArray(), function () {

			// skip invalid keys
			if ( ! patterns.validate.test(this.name) ) {
				return;
			}

			var k,
				keys = this.name.match(patterns.key),
				merge = this.value,
				reverse_key = this.name;

			while ((k = keys.pop()) !== undefined) {

				// adjust reverse_key
				reverse_key = reverse_key.replace(new RegExp("\\[" + k + "\\]$"), '');

				// push
				if ( k.match(patterns.push) ) {
					merge = self.build([], self.push_counter(reverse_key), merge);
				}

				// fixed
				else if ( k.match(patterns.fixed) ) {
					merge = self.build([], k, merge);
				}

				// named
				else if ( k.match(patterns.named) ) {
					merge = self.build({}, k, merge);
				}
			}

			json = $.extend(true, json, merge);
		});

		return json;
	};
})(jQuery);

window.TieredCommissions = window.TieredCommissions || {};

(function (window, document, $, plugin) {
	var $c = {};

	plugin.init = function () {
		plugin.cache();
		plugin.bindEvents();
		$(document).ready(function () {
			plugin.ready();
		});
	};

	plugin.ready = function () {
		$c.mainForm.parsley({
            excluded: 'input[type=button], input[type=submit], input[type=reset], input[type=hidden], [disabled], :hidden'
        });
		plugin.checkCommissionTypes();
	};

	plugin.bindEvents = function () {
		$c.addRowButton.on('click', plugin.addRowHandler);
		$c.tableData.on('click', '.remove-row', plugin.removeRowHandler);
		$c.tableData.on('change', plugin.checkCommissionTypes);
		$c.tableData.on('change', 'input, select', plugin.highlightError);
		$c.mainForm.on('submit', plugin.checkIfTableIsEmpty);
		$c.postForm.on('submit', plugin.checkIfTableIsEmpty);
	};

	plugin.cache = function () {
		$c.mainForm = $('#mainform');
		$c.postForm = $('form#post');
		$c.tableData = $('.commission-tiers');
		$c.addRowButton = $('.add-vendor-sales-commission');
		$c.selector = $('#wcv_commission_type');
		if ( $c.selector.length === 0 ) {
			$c.selector = $('#wcvendors_commission_type');
		}
	};

	plugin.randomColor = function () {
		return "#" + Math.random().toString(16).slice(2, 8).toUpperCase();
	};

	plugin.checkCommissionTypesValues = function ($ancestor, value) {
		switch (value) {
			case 'fixed':
				$ancestor.find('.percent, .fee').attr('disabled', 'disabled').attr('value',0);
				$ancestor.find('.amount').removeAttr('disabled');
				break;
			case 'fixed_fee':
				$ancestor.find('.percent').attr('disabled', 'disabled').attr('value',0);
				$ancestor.find('.amount, .fee').removeAttr('disabled');
				break;
			case 'percent':
				$ancestor.find('.amount, .fee').attr('disabled', 'disabled').attr('value',0);
				$ancestor.find('.percent').removeAttr('disabled');
				break;
			case 'percent_fee':
				$ancestor.find('.amount').attr('disabled', 'disabled').attr('value',0);
				$ancestor.find('.percent, .fee').removeAttr('disabled');
				break;
			default:
				$ancestor.find('.amount, .fee, .percent').attr('disabled', 'disabled').attr('value',0);
				break;
		}
	};

	plugin.addRowHandler = function (e) {
		e.preventDefault();
		var $parentTable = $(this).closest('.commission-tiers-table');
		var currentIndex = $parentTable.find('tr.tier-row').length;
		var key = $(this).data('key'),
			commissions_row =
				'<tr class="tier-row">' +
				'<td class="commission-field"><input type="text" name="commission_tiers[' + key + '][' + currentIndex + '][name]" placeholder="' + commission_data.name_placeholder + '" class="name" /></td>' +
				'<td class="commission-field"><select name="commission_tiers[' + key + '][' + currentIndex + '][rule]" class="rule" required><option></option>' + commission_data.rules_options + '</select></td>' +
				'<td class="commission-field"><input type="number" name="commission_tiers[' + key + '][' + currentIndex + '][value]" placeholder="' + commission_data.value_placeholder + '" class="value" required /></td>' +
				'<td class="commission-field"><select name="commission_tiers[' + key + '][' + currentIndex + '][type]" class="commission-types" required><option></option>' + commission_data.types_options + '</select></td>' +
				'<td class="commission-field"><input type="number" name="commission_tiers[' + key + '][' + currentIndex + '][amount]" placeholder="' + commission_data.amount_placeholder + '" class="amount" required/></td>' +
				'<td class="commission-field"><input type="number" name="commission_tiers[' + key + '][' + currentIndex + '][percent]" placeholder="' + commission_data.percent_placeholder + '" class="percent" min="0" max="100" step="0.01" required /></td>' +
				'<td class="commission-field"><input type="number" name="commission_tiers[' + key + '][' + currentIndex + '][fee]" placeholder="' + commission_data.fee_placeholder + '" class="fee" required /></td>' +
				'<td class="actions"><a href="#" class="remove-row" data-key="' + key + '"><svg class="wcv-icon wcv-icon-sm"><use xlink:href="' + commission_data.assets_url + 'svg/wcv-icons.svg#wcv-icon-times"></use></svg></a></td>' +
				'</tr>';

		$c.tableData.append(commissions_row);
	};

	plugin.removeRowHandler = function (e) {
		e.preventDefault();
		if ( confirm(commission_data.confirm_remove) ) {
			$(this).closest('.tier-row').remove();
		}
	};

	plugin.checkCommissionTypes = function () {
		$('.commission-types').each(function () {
			var $ancestor = $(this).closest('.tier-row');
			var value = $(this).val();

			plugin.checkCommissionTypesValues($ancestor, value);
		});
	};

	plugin.highlightError = function () {
		var $current_row = $(this).closest('.tier-row');
		var current_name = $current_row.find('.name').val();
		var current_rule = $current_row.find('.rule').val();
		var current_value = $current_row.find('.value').val();
		var current_amount = $current_row.find('.amount').val();
		var current_percent = $current_row.find('.percent').val();
		var current_fee = $current_row.find('.fee').val();

		var colors = ['#e96868', '#e74f4e', '#d9321f', '#c34242', '#b23d3c', '#a13736'];

		$('.tier-row').not($(this)).each(function () {
			var this_name = $(this).find('.name').val();
			var this_rule = $(this).find('.rule').val();
			var this_value = $(this).find('.value').val();
			var this_amount = $(this).find('.amount').val();
			var this_percent = $(this).find('.percent').val();
			var this_fee = $(this).find('.fee').val();

			if ( $current_row.index() != $(this).index() ) {
				if ( current_name == this_name ) {
					$current_row.find('.name').css('border', '1px solid ' + colors[0]).addClass('parsley-error');
					$(this).find('.name').css('border', '1px solid ' + colors[0]);
				} else {
					$current_row.find('.name').css('border', 'none').removeClass('parsley-error');
					$(this).find('.name').css('border', 'none').removeClass('parsley-error');
				}

				if ( current_value == this_value && current_rule == this_rule ) {
					$current_row.find('.value, .rule').css('border', '1px solid ' + colors[1]).addClass('parsley-error');
					$(this).find('.value, .rule').css('border', '1px solid ' + colors[1]);
					$(this).find('.value, .rule').data('parsley-error-message', commission_data.possible_duplicate);
				} else {
					$current_row.find('.value, .rule').css('border', 'none').removeClass('parsley-error');
					$(this).find('.value, .rule').css('border', 'none').removeClass('parsley-error');
				}

				if ( this_amount == 0 && this_fee == 0 && this_percent == 0 && current_amount == 0 && current_fee == 0 && current_percent == 0 ) {
					$current_row.find('.amount, .percent, .fee').css('border', '1px solid ' + colors[2]);
					$(this).find('.amount, .percent, .fee').css('border', '1px solid ' + colors[2]);
					$(this).find('.amount, .percent, .fee').data('parsley-error-message', commission_data.cant_be_zero);
				} else {
					$current_row.find('.amount, .percent, .fee').css('border', 'none').removeClass('parsley-error');
					$(this).find('.amount, .percent, .fee').css('border', 'none').removeClass('parsley-error');
				}

				if ( current_amount == this_amount && current_fee == this_fee && current_percent == this_percent ) {
					$current_row.find('.amount, .percent, .fee').css('border', '1px solid ' + colors[3]);
					$(this).find('.amount, .percent, .fee').css('border', '1px solid ' + colors[3]);
					$(this).find('.amount, .percent, .fee').data('parsley-error-message', commission_data.conflicting_rows);
				} else {
					$current_row.find('.amount, .percent, .fee').css('border', 'none').removeClass('parsley-error');
					$(this).find('.amount, .percent, .fee').css('border', 'none').removeClass('parsley-error');
				}
			}
		});
	};

	plugin.checkIfTableIsEmpty = function (e) {
		var selectedType = $c.selector.val();
		var formData = $(this).serializeObject();
		var commissionTypes = ['vendor_sales', 'product_sales', 'product_price'];

		if ( ! selectedType )
			return;

		if ( - 1 === commissionTypes.indexOf(selectedType) )
			return;

		console.log(selectedType, formData);
		if (
			! formData.hasOwnProperty('commission_tiers')
			|| ! formData.commission_tiers.hasOwnProperty(selectedType)
			|| formData.commission_tiers[selectedType].length === 0
		) {
			e.preventDefault();
			alert(commission_data.cant_be_empty);
		}
	};

	$(plugin.init);
}(window, document, jQuery, window.TieredCommissions));
