<?php
/**
 * WooCommerceKlaviyo API
 *
 * Handles WCK-API endpoint requests
 *
 * @author      Klaviyo
 * @category    API
 * @package     WooCommerceKlaviyo/API
 * @since       2.0
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class WCK_API {
  const VERSION = '2.1.6';

  const KLAVIYO_BASE_URL = 'klaviyo/v1/';
  const ORDERS_ENDPOINT = 'orders';
  const EXTENSION_VERSION_ENDPOINT = 'version';
  const PRODUCTS_ENDPOINT = 'products';

  const DEFAULT_RECORDS_PER_PAGE = '50';
  const DATE_MODIFIED = 'post_modified_gmt';
  const POST_STATUS_ANY = 'any';
  
  const INVALID_KEYS = 'invalid consumer_key or consumer_secret';
  const INVALID_TOKEN = 'invalid token';
}

function get_orders_count(WP_REST_Request $request) {
  if (!is_valid_request($request)) {
    return array('error' => WCK_API::INVALID_TOKEN);
  }

  $args = process_resource_args($request, 'shop_order');

  $loop = new WP_Query($args);
  $order_ids = array();
  while ($loop->have_posts()) {
    $loop->the_post();
    $order_id = get_the_ID();
    array_push($order_ids, $order_id);
  }
  return array('order_count' => $loop->found_posts);
}

function get_products_count(WP_REST_Request $request) {
  if (!is_valid_request($request)) {
    return array('error' => WCK_API::INVALID_TOKEN);
  }

  $args = process_resource_args($request, 'product');

  $loop = new WP_Query($args);
  $product_ids = array();
  while ($loop->have_posts()) {
    $loop->the_post();
    $product_id = get_the_ID();
    array_push($product_ids, $product_id);
  }
  return array('product_count' => $loop->found_posts);
}

function get_products(WP_REST_Request $request) {
  if (!is_valid_request($request)) {
    return array('error' => WCK_API::INVALID_TOKEN);
  }

  $args = process_resource_args($request, 'product');

  $loop = new WP_Query($args);
  $product_ids = array();
  while ($loop->have_posts()) {
    $loop->the_post();
    $product_id = get_the_ID();
    array_push($product_ids, $product_id);
  }
  $data = array('product_ids' => $product_ids);
  return $data;
}

function get_extension_version($data) {
  return array('version' => WCK_API::VERSION);
}

function is_valid_request($request) {
  $token = $request->get_param('token');
  $klaviyo_settings = get_option('klaviyo_settings');
  if (!empty($klaviyo_settings['public_api_key'])) {
    if ($token == base64_encode($klaviyo_settings['public_api_key'])) {
      return true;
    }
  }
  return false;
}

function process_resource_args($request, $post_type) {
  $page_limit = $request->get_param('page_limit');
  if (empty($page_limit)) {
    $page_limit = WCK_API::DEFAULT_RECORDS_PER_PAGE;
  }
  $date_modified_after = $request->get_param('date_modified_after');
  $date_modified_before = $request->get_param('date_modified_before');
  $page = $request->get_param('page');

  $args = array (
    'post_type'   => $post_type,
    'posts_per_page'    => $page_limit,
    'post_status'    => WCK_API::POST_STATUS_ANY,
    'paged'      => $page,
    'date_query' => array(
      array(
        'column' => WCK_API::DATE_MODIFIED,
        'after'  => $date_modified_after,
        'before' => $date_modified_before
        )
      ),
    );
  return $args;
}

function get_orders(WP_REST_Request $request) {
  if (!is_valid_request($request)) {
    return array('error' => WCK_API::INVALID_TOKEN);
  }

  $args = process_resource_args($request, 'shop_order');

  $loop = new WP_Query($args);
  $order_ids = array();
  while ($loop->have_posts()) {
    $loop->the_post();
    $order_id = get_the_ID();
    array_push($order_ids, $order_id);
  }
  $data = array('order_ids' => $order_ids);
  return $data;
}

add_action( 'rest_api_init', function() {
  register_rest_route(WCK_API::KLAVIYO_BASE_URL, WCK_API::EXTENSION_VERSION_ENDPOINT, array(
    'methods' => WP_REST_Server::READABLE,
    'callback' => 'get_extension_version',
    )
  );
});

add_action( 'rest_api_init', function() {
  register_rest_route(WCK_API::KLAVIYO_BASE_URL, 'orders/count', array(
    'methods' => WP_REST_Server::READABLE,
    'callback' => 'get_orders_count',
    )
  );
});

add_action( 'rest_api_init', function() {
  register_rest_route(WCK_API::KLAVIYO_BASE_URL, 'products/count', array(
    'methods' => WP_REST_Server::READABLE,
    'callback' => 'get_products_count',
    )
  );
});

add_action( 'rest_api_init', function() {
  register_rest_route(WCK_API::KLAVIYO_BASE_URL, WCK_API::ORDERS_ENDPOINT, array(
    'methods' => WP_REST_Server::READABLE,
    'callback' => 'get_orders',
    'args' => array(
      'id' => array(
        'validate_callback' => 'is_numeric'
      ),
    ),
    'permission_callback' => function() {
      return true;
    })
  );
});

add_action( 'rest_api_init', function() {
  register_rest_route(WCK_API::KLAVIYO_BASE_URL, WCK_API::PRODUCTS_ENDPOINT, array(
    'methods' => WP_REST_Server::READABLE,
    'callback' => 'get_products',
    'args' => array(
      'id' => array(
        'validate_callback' => 'is_numeric'
      ),
    ),
    'permission_callback' => function() {
      return true;
    })
  );
});