<?php

/**
 * Defined Post Types
 * 
 *      define('RML_TYPE_FOLDER', 0);
 *      define('RML_TYPE_COLLECTION', 1);
 *      define('RML_TYPE_GALLERY', 2);
 *
 * Example Szenario #1:
 *   1. User navigates to http://example.com/rml/collection1
 *   2. Use wp_rml_get_by_absolute_path("/collection1") to get the RML_Folder Object
 *   3. (Additional check) $folder->is(RML_TYPE_COLLECTION) to check, if it is a collection.
 *   4. Iterate the childrens with foreach ($folder->children as $value) { }
 *   5. In collection can only be other collections or galleries.
 * 
 *   6. (Additional check) $value->is(RML_TYPE_GALLERY) to check, if it is a gallery.
 *   7. Fetch the IDs with $value->fetchFileIds();
 * 
 * 
 * If you want to use more attractive functions look into the RML_Structure Class.
 * You easily get it with RML_Structure::getInstance() (Singleton).
 * 
 * Meaning: Root = Unorganized Pictures
 * 
 * 
 * Filters and actions:
 *      RML/Folder/QueryArgs
 *      RML/Folder/QueryResult
 *      RML/Folder/QueryCountArgs
 *      RML/Folder/Deleted
 *      RML/Folder/Created
 *      RML/Folder/Renamed
 *      RML/Folder/Moved
 *      RML/Item/Moved
 *      RML/Item/MovedFinish
 * 
 * apply_filters('RML/Folder/QueryArgs', $args)
 *      $args           Arguments for read query WP_Query
 * 
 * apply_filters('RML/Folder/QueryResult', $posts)
 *      $posts          Result of WP_Query
 * 
 * apply_filters('RML/Folder/QueryCountArgs', $args)
 *      $args           Arguments for count query RML_WP_Query_Count class
 * 
 * do_action("RML/Folder/Deleted", $id, $posts)
 *      $id             ID of the deleted folder
 *      $posts          WP_Query result of folders content
 * 
 * do_action("RML/Folder/Created", $parent, $name, $type)
 *      $parent         Parent id
 *      $name           New name of the folder
 *      $type           RML_TYPE_... constant
 * 
 * do_action('RML/Folder/Renamed', $name, $obj)
 *      $name           New name of the folder
 *      $obj            Folder object RML_Folder
 * 
 * do_action('RML/Folder/Moved', $obj, $id, $ord, $force)
 *      $obj            Folder object RML_Folder
 *      $id             ID of the moved folder
 *      $ord            Order number of the moved folder
 *      $force          Boolean
 * 
 * do_action("RML/Item/Moved", $value, $obj)
 *      $value          Post ID of attachment
 *      $obj            Folder object RML_Folder or null for root
 * 
 * do_action("RML/Item/MovedFinish", $folder, $ids);
 *      $folder         Folder object RML_Folder or null for root
 *      $ids            Array of moved post ids
 */

if (!function_exists('wp_rml_root_childs')) {
    /**
     * Gets the first level childs of the media library.
     * 
     * @return Array of RML_Folder objects
     */
    function wp_rml_root_childs() {
        return RML_Structure::getInstance()->getTree();
    }
}

if (!function_exists('wp_rml_select_tree')) {
    /**
     * Returns a .rml-root-list with an given tree. The selected folder id is
     * saved automatically in a hidden input type.
     * 
     * @param inputName the name for the hidden input type
     * @param selected the selected folder id (saved also in hidden input type)
     * @param tree Array of RML_Folder objects
     * @param extraClasses classes for the rml root list container
     * @return Formatted HTML string
     * 
     * Experimental:
     * <strong>Note #1</strong> The select tree has a javascript callback when it
     * is initalized. You can bind it with this snippet:
     * 
     * window.rml.hooks.register("customList", function(e, obj) {
     *       //if (obj.hasClass("my-extra-class")) {
     *            alert(obj.html());
     *       //}
     * });
     * 
     * <strong>Note #2</strong> If you want to use the select tree after a DOM change (ajax,
     * for example: Modal dialog in visual editor) please call the javascript function
     * window.rml.library.customLists() to affect the initalization referred to Note #1.
     * 
     * <strong>Note #3</strong> You can use a sub class of RML_Folder to customize your tree.
     * 
     * @see To see an demo how to use it, have a look at ../inc/admin_footer/sidebar.dummy.php:96
     * @see ../assets/js/library.js:customLists()
     */
    function wp_rml_select_tree($inputName, $selected, $tree, $extraClasses = "") {
        $output = '<div class="rml-root-list rml-custom-list ' . $extraClasses . '">
                <input type="hidden" name="' . $inputName . '" value="' . $selected . '" />';
        
        $output .= RML_Structure::getInstance()->getView()->treeHTML($selected, $tree);
        $output .= '</div>';
        return $output;
    }
}

if (!function_exists('wp_rml_create')) {
    /**
     * Creates a new folder.
     * 
     * @selfexplanatory
     */
    function wp_rml_create($name, $parent, $type) {
        return RML_Structure::getInstance()->createFolder($name, $parent, $type);
    }
}

if (!function_exists('wp_rml_rename')) {
    /**
     * Renames a folder by ID.
     * 
     * @selfexplanatory
     */
    function wp_rml_rename($name, $id) {
        return RML_Structure::getInstance()->renameFolder($name, $id);
    }
}

if (!function_exists('wp_rml_delete')) {
    /**
     * Deletes a folder by ID.
     * 
     * @selfexplanatory
     */
    function wp_rml_delete($id) {
        return RML_Structure::getInstance()->deleteFolder($id);
    }
}

if (!function_exists('wp_rml_move')) {
    /**
     * Moves a set of attachments to a specific folder.
     * 
     * @param to Folder ID
     * @param ids Array of attachment ids
     */
    function wp_rml_move($to, $ids) {
        if (!is_array($ids) || count($ids) == 0 || $to == null) {
                return;
        }
        
        $folder = RML_Structure::getInstance()->getFolderById($to);
        if ($folder !== null) {
            $folder->moveItemsHere($ids);
            do_action("RML/Item/MovedFinish", $folder, $ids);
        }else{
            if (is_array($ids) && count($ids) > 0) {
                foreach ($ids as $value) {
                    // Check if other fails are counted
    	            $errors = apply_filters("RML/Item/ValidateMove", array(), $ids, -1);
    	            if (count($errors) > 0) {
    	                continue;
    	            }
                    
                    update_post_meta($value, "_rml_folder", -1);
                    do_action("RML/Item/Moved", $value, null);
                }
            }
            do_action("RML/Item/MovedFinish", null, $ids);
        }
    }
}

if (!function_exists('wp_rml_dropdown')) {
    /**
     * This functions returns a HTML formatted string which contains
     * <options> elements with all folders, collections and galleries.
     * 
     * @param $selected The selected item
     *              "": "All Files"
     *              -1: "Root"
     *              int: Folder ID
     * @param $disabled array Defines, which folder types are disabled (@see ./real-media-library.php for Constant-Types)
     *                        Default disabled is RML_TYPE_COLLECTION
     * @param $useAll boolean Defines, if "All Files" should be showed
     * @return String
     */
    function wp_rml_dropdown($selected, $disabled, $useAll = true) {
        return RML_Structure::getInstance()->optionsFasade($selected, $disabled, $useAll);
    }
}

if (!function_exists('wp_rml_dropdown_collection')) {
    /**
     * This functions returns a HTML formatted string which contains
     * <options> elements with all folders, collections and galleries.
     * Note: Only COLLECTIONS are SELECTABLE!
     * 
     * @param $selected The selected item
     *              "": "All Files"
     *              -1: "Root"
     *              int: Folder ID
     * @return String
     */
    function wp_rml_dropdown_collection($selected) {
        return wp_rml_dropdown($selected, array(0,2,3,4));
    }
}

if (!function_exists('wp_rml_dropdown_gallery')) {
    /**
     * This functions returns a HTML formatted string which contains
     * <options> elements with all folders, collections and galleries.
     * Note: Only GALLERIES are SELECTABLE!
     * 
     * @param $selected The selected item
     *              "": "All Files"
     *              -1: "Root"
     *              int: Folder ID
     * @return String
     */
    function wp_rml_dropdown_gallery($selected) {
        return wp_rml_dropdown($selected, array(0,1,3,4));
    }
}

if (!function_exists('wp_rml_dropdown_gallery_or_collection')) {
    /**
     * This functions returns a HTML formatted string which contains
     * <options> elements with all folders, collections and galleries.
     * Note: Only GALLERIES AND COLLECTIONS are SELECTABLE!
     * 
     * @param $selected The selected item
     *              "": "All Files"
     *              -1: "Root"
     *              int: Folder ID
     * @return String
     */
    function wp_rml_dropdown_gallery_or_collection($selected) {
        return wp_rml_dropdown($selected, array(0,3,4));
    }
}

if (!function_exists('wp_rml_is_type')) {
    /**
     * Determines, if a Folder is a special folder type.
     * 
     * @param $folder RML_Folder
     * @param $allowed array Defines, which folder types are allowed (@see ./real-media-library.php for Constant-Types) 
     * @return boolean
     */
    function wp_rml_is_type($folder, $allowed) {
        if (!$folder instanceof RML_Folder) {
            return false;
        }
        
        return in_array($folder->type, $allowed);
    }
}

if (!function_exists('wp_rml_get_by_id')) {
    /**
     * This functions checks if a specific folder exists by ID and is
     * a given allowed RML Folder Type. If the given folder is -1 you will
     * get the first level folders.
     * 
     * @param $id int Folder ID
     * @param $allowed array Defines, which folder types are allowed (@see ./real-media-library.php for Constant-Types)
     *                       If it is null, all folder types are allowed.
     * @param $mustBeFolderObject Defines if the function may return the wp_rml_root_childs result
     * @return RML_Folder object or NULL
     * 
     * Note: The Folder ID must be a valid Folder ID, not Root and "All Files" => FolderID > -1
     */
    function wp_rml_get_by_id($id, $allowed = null, $mustBeFolderObject = false) {
        if (!is_numeric($id)) {
            return null;
        }
        
        if ($mustBeFolderObject == false && ($id == -1 || $id == "-1")) {
            return wp_rml_root_childs();
        }
        
        $folder = RML_Structure::getInstance()->getFolderByID($id);
        
        if (is_array($allowed)) {
            if (!wp_rml_is_type($folder, $allowed)) {
                return null;
            }
        }
        
        return $folder;
    }
}

if (!function_exists('wp_rml_get_by_absolute_path')) {
    /**
     * This functions checks if a specific folder exists by absolute path and is
     * a given allowed RML Folder Type.
     * 
     * @param $path string Folder Absolute Path
     * @param $allowed array Defines, which folder types are allowed (@see ./real-media-library.php for Constant-Types)
     *                       If it is null, all folder types are allowed.
     * @return RML_Folder object or NULL
     * 
     * Note: The absolute path may not be "/" (Root).
     */
    function wp_rml_get_by_absolute_path($path, $allowed = null) {
        $folder = RML_Structure::getInstance()->getFolderByAbsolutePath($path);
        
        if (is_array($allowed)) {
            if (!wp_rml_is_type($folder, $allowed)) {
                return null;
            }
        }
        
        return $folder;
    }
}

if (!function_exists('wp_rml_test_showcase')) {
    /**
     * Outputs a few options for the api usage
     */
    function wp_rml_test_showcase() {
        echo '<br /><br />
                Selected: Root; All folder types allowed; "All Files" disabled
                <select>
                        ' . wp_rml_dropdown(-1, array(), false) . '
                </select>';
                
        echo '<br /><br />
                Selected: All Files; Only folders allowed; "All Files" allowed
                <select>
                        ' . wp_rml_dropdown(-1, array(RML_TYPE_COLLECTION, RML_TYPE_GALLERY), true) . '
                </select>';
                
        echo '<br /><br />
                Select a collection
                <select>
                        ' . wp_rml_dropdown_collection("") . '
                </select>';
                
        echo '<br /><br />
                Select a gallery
                <select>
                        ' . wp_rml_dropdown_gallery("") . '
                </select>';
                
        echo '<br /><br />
                Select a gallery or collection
                <select>
                        ' . wp_rml_dropdown_gallery_or_collection("") . '
                </select>';
                
        echo '<br /><br />
                Get Folder with childrens by Absolute Path
        ';
        RML_Core::print_r(wp_rml_get_by_absolute_path("/others/from-you/2016"));
        
        RML_Core::print_r(wp_rml_root_childs());
        
        echo '<br /><br />
                Check if root has child folder
        ';
        var_dump(RML_Structure::getInstance()->hasChildSlug(-1, "others", false));
    }
}

?>