<?php
defined( 'ABSPATH' ) or die( 'No script kiddies please!' );

class RML_Ajax {
	private static $me = null;
        
    private function __construct() {
            
    }
    
    /**
     * Checks, if the permission to use a specific AJAX 
     * request is given. It automatically dies the current
     * screen and prints out an error.
     * 
     * @param nonce The nonce to check
     * @param cap The needed capability
     * @private
     */
    private function check($nonce = false, $cap = "upload_files") {
        if ($nonce !== false) {
            check_ajax_referer($nonce, 'nonce');
        }
        
        if (!current_user_can($cap)) {
            wp_send_json_error(__("Something went wrong."));
        }
    }
    
    /**
     * Wipes the RML settings. That means: Attachment relations to
     * the folders and the folders.
     * 
     * @REQUEST method 'all' or 'rel'
     */
    public function wp_ajax_wipe() {
        // Security checks
        $this->check('rmlAjaxWipe', 'manage_options');
        
        // Process
        global $wpdb;
        $table_name = RML_Core::getInstance()->getTableName();
        
        $sqlMeta = "UPDATE $wpdb->postmeta SET meta_value = -1 WHERE meta_key = '_rml_folder'";
        $sqlFolders = "DELETE FROM $table_name";
        
        $method = $_REQUEST["method"];
        if ($method == "all") {
            $wpdb->query($sqlMeta);
            $wpdb->query($sqlFolders);
        }else if ($method == "rel") {
            $wpdb->query($sqlMeta);
        }

        wp_send_json_success();
    }
    
    /**
     * Creates a folder.
     * 
     * @POST name The name of the folder
     * @POST parent The ID of the parent folder, use -1 for Root level
     * @POST type The type of the folder (see /real-media-library.php contants)
     */
    public function wp_ajax_folder_create() {
        // Security checks
        $this->check('rmlAjaxFolderCreate');
        
        // Process
        $name = isset($_POST["name"]) ? $_POST["name"] : "";
        $parent = isset($_POST["parent"]) ? $_POST["parent"] : -1;
        $type = isset($_POST["type"]) ? $_POST["type"] : -1;
        
        $result = wp_rml_create($name, $parent, $type);
        
        if ($result === true) {
            wp_send_json_success();
        }else if (is_array($result)) {
            wp_send_json_error(implode(" ", $result));
        }else{
            wp_send_json_error(__("Please use a valid folder name and make sure, there is no duplicate folder name.", RML_TD));
        }
    }
    
    /**
     * Renames a folder.
     * 
     * @POST name The new name of the folder
     * @POST id The folder id which should be renamed
     */
    public function wp_ajax_folder_rename() {
        // Security checks
        $this->check('rmlAjaxFolderRename');
        
        // Process
        $name = isset($_POST["name"]) ? $_POST["name"] : "";
        $id = isset($_POST["id"]) ? $_POST["id"] : -1;
        
        if (($folder = wp_rml_rename($name, $id)) !== false) {
            wp_send_json_success(array(
                "slug" => $folder->absolutePath()
            ));
        }else{
            wp_send_json_error(__("Please use a valid folder name and make sure, there is no duplicate folder name.", RML_TD));
        }
    }
    
    /**
     * Deletes a folder.
     * 
     * @POST id The folder id
     */
    public function wp_ajax_folder_delete() {
        // Security checks
        $this->check('rmlAjaxFolderDelete');
        
        // Process
        $id = isset($_POST["id"]) ? $_POST["id"] : -1;
        
        $result = wp_rml_delete($id);
        
        if ($result === true) {
            wp_send_json_success();
        }else if (is_array($result)) {
            wp_send_json_error(implode(" ", $result));
        }else{
            wp_send_json_error("Something went wrong.");
        }
    }
    
    /**
     * Moves one or more attachments to a given folder.
     * 
     * @POST ids (array) One or more attachment ids
     * @POST to The folder id
     */
    public function wp_ajax_bulk_move() {
        // Security checks
        $this->check('rmlAjaxBulkMove');
        
        // Process
        $ids = isset($_POST["ids"]) ? $_POST["ids"] : null;
        $to = isset($_POST["to"]) ? $_POST["to"] : null;
        
        wp_rml_move($to, $ids);
        
        wp_die();
    }
    
    /**
     * Sorts the folder tree.
     * 
     * @POST ids (array) Serialized array of the new tree structure
     */
    public function wp_ajax_bulk_sort() {
        // Security checks
        $this->check('rmlAjaxBulkSort');
        
        // Process
        $ids = isset($_POST["ids"]) ? $_POST["ids"] : null;
        if (!is_array($ids) || count($ids) == 0) {
                wp_send_json_error(__("Something went wrong."));
        }
        
        // fid (folderid): pid: (parentid)
        $struct = RML_Structure::getInstance();
        
        $changer = array(); // This folders can be changed
        
        // Check, if types are right
        $i = 0;
        $foundError = false;
        foreach ($ids as $value) {
            $fid = $value["fid"]; // Folder ID
            $pid = $value["pid"]; // Parent ID
            
            // Check
            if (!is_numeric($fid) || !is_numeric($pid)) {
                    continue;
            }
            
            // Execute
            $fid = $struct->getFolderById($fid);
            if ($fid !== null && $struct->isAllowedTo($pid, $fid->type)) {
                    // Check, if parent may have this name as folder.
                    
                    $changer[] = array($fid, $pid, $i);
            }else{
                    $foundError = true;
                    break;
            }
            
            $i++;
        }
        
        // Change it!
        if ($foundError) {
            wp_send_json_error(__("Something went wrong. Please be sure folders can not be in collections and galleries, collections can only be in folders and other collections and galleries can only be in collections.", RML_TD));
        }else{
            foreach ($changer as $value) {
                $value[0]->setParent($value[1], $value[2]);
            }
            do_action("RML/Structure/Rebuild");
            wp_send_json_success();
        }
    }
    
    /**
     * Get the current folder count of one or more folders.
     * 
     * @REQUEST ids (array|string) Array or imploded (,) string of folder ids
     *                             Use ALL for the all files count
     */
    public function wp_ajax_folder_count() {
        // Security checks
        $this->check('rmlAjaxFolderCount');
        
        // Process
        $ids = isset($_REQUEST["ids"]) ? $_REQUEST["ids"] : "";
        if (!is_array($ids)) {
            $ids = explode(',', $ids);
        }
        
        if (is_array($ids) && count($ids) > 0) {
            $ids = array_unique($ids);
            $result = array();

            foreach ($ids as $folderId) {
                $folderId = trim($folderId);

                if (is_numeric($folderId) || $folderId == "" || $folderId == "ALL") {
                    if ($folderId == "-1") {
                        $result["-1"] = RML_Structure::getInstance()->getCntRoot();
                    }else{
                        // ALL is the constant for all Files
                        if ($folderId == "ALL") {
                            $folderId = "";
                        }
                        
                        $query = new RML_WP_Query_Count(
                            apply_filters('RML/Folder/QueryCountArgs', array(
                            	'post_status' => 'inherit',
                            	'post_type' => 'attachment',
                            	'rml_folder' => $folderId
                            ))
                        );
                        
                        // Reset it to ALL
                        if ($folderId == "") {
                            $folderId = "ALL";
                        }
                        
                        if (isset($query->posts[0])) {
                            $result[$folderId] = $query->posts[0];
                        }else{
                            $result[$folderId] = 0;
                        }
                    }
                }
            }
            wp_send_json_success($result);
        }else{
            wp_send_json_error(__('Something went wrong.', RML_TD));
        }
    }
    
    /**
     * Save the size of the resized sidebar so the sidebar.dummy.php
     * can modify the CSS.
     * 
     * @POST width The new width of the sidebar
     */
    public function wp_ajax_sidebar_resize() {
        // Security checks
        $this->check('rmlAjaxSidebarResize');
        
        // Process
        $width = isset($_POST["width"]) ? $_POST["width"] : 0;
        
        if ($width > 0) {
            setcookie( "rml_" . get_current_blog_id() . "_resize", $width, strtotime( '+365 days' ), '/' );
            wp_send_json_success();
        }else{
            wp_send_json_error();
        }
    }
    
    public static function getInstance() {
        if (self::$me == null) {
                self::$me = new RML_Ajax();
        }
        return self::$me;
    }
}

?>